% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-ou-fixed.R
\name{SimSSMOUFixed}
\alias{SimSSMOUFixed}
\title{Simulate Data from the
Ornstein-Uhlenbeck Model
using a State Space Model Parameterization
(Fixed Parameters)}
\usage{
SimSSMOUFixed(
  n,
  time,
  delta_t = 1,
  mu0,
  sigma0_l,
  mu,
  phi,
  sigma_l,
  nu,
  lambda,
  theta_l,
  type = 0,
  x = NULL,
  gamma = NULL,
  kappa = NULL
)
}
\arguments{
\item{n}{Positive integer.
Number of individuals.}

\item{time}{Positive integer.
Number of time points.}

\item{delta_t}{Numeric.
Time interval
(\eqn{\Delta_t}).}

\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(sigma0))})
of the covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{mu}{Numeric vector.
The long-term mean or equilibrium level
(\eqn{\boldsymbol{\mu}}).}

\item{phi}{Numeric matrix.
The drift matrix
which represents the rate of change of the solution
in the absence of any random fluctuations
(\eqn{\boldsymbol{\Phi}}).
It also represents the rate of mean reversion,
determining how quickly the variable returns to its mean.}

\item{sigma_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(sigma))})
of the covariance matrix of volatility
or randomness in the process
(\eqn{\boldsymbol{\Sigma}}).}

\item{nu}{Numeric vector.
Vector of intercept values for the measurement model
(\eqn{\boldsymbol{\nu}}).}

\item{lambda}{Numeric matrix.
Factor loading matrix linking the latent variables
to the observed variables
(\eqn{\boldsymbol{\Lambda}}).}

\item{theta_l}{Numeric matrix.
Cholesky factorization (\code{t(chol(theta))})
of the covariance matrix
of the measurement error
(\eqn{\boldsymbol{\Theta}}).}

\item{type}{Integer.
State space model type.
See Details for more information.}

\item{x}{List.
Each element of the list is a matrix of covariates
for each individual \code{i} in \code{n}.
The number of columns in each matrix
should be equal to \code{time}.}

\item{gamma}{Numeric matrix.
Matrix linking the covariates to the latent variables
at current time point
(\eqn{\boldsymbol{\Gamma}}).}

\item{kappa}{Numeric matrix.
Matrix linking the covariates to the observed variables
at current time point
(\eqn{\boldsymbol{\kappa}}).}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
Each element of \code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ID numbers with length \code{l},
where \code{l} is the value of the function argument \code{time}.
\item \code{time}: A vector time points of length \code{l}.
\item \code{y}: A \code{l} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{l} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{l} by \code{j} matrix of values for the covariates
(when covariates are included).
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from the
Ornstein-Uhlenbeck (OU) model
using a state space model parameterization.
It assumes that the parameters remain constant
across individuals and over time.
}
\details{
\subsection{Type 0}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\mathbf{y}_{i, t}},
\eqn{\boldsymbol{\eta}_{i, t}},
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
are random variables
and
\eqn{\boldsymbol{\nu}},
\eqn{\boldsymbol{\Lambda}},
and
\eqn{\boldsymbol{\Theta}}
are model parameters.
\eqn{\mathbf{y}_{i, t}}
represents a vector of observed random variables,
\eqn{\boldsymbol{\eta}_{i, t}}
a vector of latent random variables,
and
\eqn{\boldsymbol{\varepsilon}_{i, t}}
a vector of random measurement errors,
at time \eqn{t} and individual \eqn{i}.
\eqn{\boldsymbol{\nu}}
denotes a vector of intercepts,
\eqn{\boldsymbol{\Lambda}}
a matrix of factor loadings,
and
\eqn{\boldsymbol{\Theta}}
the covariance matrix of
\eqn{\boldsymbol{\varepsilon}}.

An alternative representation of the measurement error
is given by
\deqn{
    \boldsymbol{\varepsilon}_{i, t}
    =
    \boldsymbol{\Theta}^{\frac{1}{2}}
    \mathbf{z}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \mathbf{z}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \mathbf{I}
    \right)
  }
where
\eqn{\mathbf{z}_{i, t}} is a vector of
independent standard normal random variables and
\eqn{
    \left( \boldsymbol{\Theta}^{\frac{1}{2}} \right)
    \left( \boldsymbol{\Theta}^{\frac{1}{2}} \right)^{\prime}
    =
    \boldsymbol{\Theta} .
  }

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\eta}_{i, t}
    -
    \boldsymbol{\mu}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t}
  }
where
\eqn{\boldsymbol{\mu}}
is the long-term mean or equilibrium level,
\eqn{\boldsymbol{\Phi}}
is the rate of mean reversion,
determining how quickly the variable returns to its mean,
\eqn{\boldsymbol{\Sigma}}
is the matrix of volatility
or randomness in the process, and
\eqn{\mathrm{d}\boldsymbol{W}}
is a Wiener process or Brownian motion,
which represents random fluctuations.
}

\subsection{Type 1}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right) .
  }

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\eta}_{i, t}
    -
    \boldsymbol{\mu}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t}
  }
where
\eqn{\mathbf{x}_{i, t}} represents a vector of covariates
at time \eqn{t} and individual \eqn{i},
and \eqn{\boldsymbol{\Gamma}} the coefficient matrix
linking the covariates to the latent variables.
}

\subsection{Type 2}{

The measurement model is given by
\deqn{
    \mathbf{y}_{i, t}
    =
    \boldsymbol{\nu}
    +
    \boldsymbol{\Lambda}
    \boldsymbol{\eta}_{i, t}
    +
    \boldsymbol{\kappa}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\varepsilon}_{i, t},
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\varepsilon}_{i, t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Theta}
    \right)
  }
where
\eqn{\boldsymbol{\kappa}} represents the coefficient matrix
linking the covariates to the observed variables.

The dynamic structure is given by
\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \boldsymbol{\Phi}
    \left(
    \boldsymbol{\eta}_{i, t}
    -
    \boldsymbol{\mu}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Gamma}
    \mathbf{x}_{i, t}
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t} .
  }
}

\subsection{The OU model as a linear stochastic differential equation model}{

The OU model is a first-order
linear stochastic differential equation model
in the form of

\deqn{
    \mathrm{d} \boldsymbol{\eta}_{i, t}
    =
    \left(
    \boldsymbol{\iota}
    +
    \boldsymbol{\Phi}
    \boldsymbol{\eta}_{i, t}
    \right)
    \mathrm{d}t
    +
    \boldsymbol{\Sigma}^{\frac{1}{2}}
    \mathrm{d}
    \mathbf{W}_{i, t}
  }
where
\eqn{\boldsymbol{\mu} = - \boldsymbol{\Phi}^{-1} \boldsymbol{\iota}}
and, equivalently
\eqn{\boldsymbol{\iota} = - \boldsymbol{\Phi} \boldsymbol{\mu}}.
}
}
\examples{
# prepare parameters
set.seed(42)
## number of individuals
n <- 5
## time points
time <- 50
delta_t <- 0.10
## dynamic structure
p <- 2
mu0 <- c(-3.0, 1.5)
sigma0 <- 0.001 * diag(p)
sigma0_l <- t(chol(sigma0))
mu <- c(5.76, 5.18)
phi <- matrix(
  data = c(
    -0.10,
    0.05,
    0.05,
    -0.10
  ),
  nrow = p
)
sigma <- matrix(
  data = c(
    2.79,
    0.06,
    0.06,
    3.27
  ),
  nrow = p
)
sigma_l <- t(chol(sigma))
## measurement model
k <- 2
nu <- rep(x = 0, times = k)
lambda <- diag(k)
theta <- 0.001 * diag(k)
theta_l <- t(chol(theta))
## covariates
j <- 2
x <- lapply(
  X = seq_len(n),
  FUN = function(i) {
    matrix(
      data = stats::rnorm(n = time * j),
      nrow = j,
      ncol = time
    )
  }
)
gamma <- diag(x = 0.10, nrow = p, ncol = j)
kappa <- diag(x = 0.10, nrow = k, ncol = j)

# Type 0
ssm <- SimSSMOUFixed(
  n = n,
  time = time,
  delta_t = delta_t,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  mu = mu,
  phi = phi,
  sigma_l = sigma_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 0
)

plot(ssm)

# Type 1
ssm <- SimSSMOUFixed(
  n = n,
  time = time,
  delta_t = delta_t,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  mu = mu,
  phi = phi,
  sigma_l = sigma_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 1,
  x = x,
  gamma = gamma
)

plot(ssm)

# Type 2
ssm <- SimSSMOUFixed(
  n = n,
  time = time,
  delta_t = delta_t,
  mu0 = mu0,
  sigma0_l = sigma0_l,
  mu = mu,
  phi = phi,
  sigma_l = sigma_l,
  nu = nu,
  lambda = lambda,
  theta_l = theta_l,
  type = 2,
  x = x,
  gamma = gamma,
  kappa = kappa
)

plot(ssm)

}
\references{
Chow, S.-M., Ho, M. R., Hamaker, E. L., & Dolan, C. V. (2010).
Equivalence and differences between structural equation modeling
and state-space modeling techniques.
\emph{Structural Equation Modeling: A Multidisciplinary Journal},
17(2), 303-332.
\doi{10.1080/10705511003661553}

Chow, S.-M., Losardo, D., Park, J., & Molenaar, P. C. M. (2023).
Continuous-time dynamic models:
Connections to structural equation models and other discrete-time models.
In R. H. Hoyle (Ed.),
Handbook of structural equation modeling (2nd ed.).
The Guilford Press.

Harvey, A. C. (1990).
Forecasting, structural time series models and the Kalman filter.
Cambridge University Press.
\doi{10.1017/cbo9781107049994}

Oravecz, Z., Tuerlinckx, F., & Vandekerckhove, J. (2011).
A hierarchical latent stochastic differential equation model
for affective dynamics.
Psychological Methods,
16 (4), 468-490.
\doi{10.1037/a0024375}

Uhlenbeck, G. E., & Ornstein, L. S. (1930).
On the theory of the brownian motion.
Physical Review,
36 (5), 823-841.
\doi{10.1103/physrev.36.823}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{LinSDE2SSM}()},
\code{\link{LinSDECovEta}()},
\code{\link{LinSDECovY}()},
\code{\link{LinSDEMeanEta}()},
\code{\link{LinSDEMeanY}()},
\code{\link{ProjectToHurwitz}()},
\code{\link{ProjectToStability}()},
\code{\link{SSMCovEta}()},
\code{\link{SSMCovY}()},
\code{\link{SSMInterceptEta}()},
\code{\link{SSMInterceptY}()},
\code{\link{SSMMeanEta}()},
\code{\link{SSMMeanY}()},
\code{\link{SimAlphaN}()},
\code{\link{SimBetaN}()},
\code{\link{SimBetaN2}()},
\code{\link{SimBetaNCovariate}()},
\code{\link{SimCovDiagN}()},
\code{\link{SimCovN}()},
\code{\link{SimIotaN}()},
\code{\link{SimNuN}()},
\code{\link{SimPhiN}()},
\code{\link{SimPhiN2}()},
\code{\link{SimPhiNCovariate}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinSDEFixed}()},
\code{\link{SimSSMLinSDEIVary}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{SpectralRadius}()},
\code{\link{TestPhi}()},
\code{\link{TestPhiHurwitz}()},
\code{\link{TestStability}()},
\code{\link{TestStationarity}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Simulation of State Space Models Data Functions}
\keyword{ou}
\keyword{sim}
\keyword{simStateSpace}
