#' Detect reversion mutations
#' 
#' @description getReversions() detects reversion mutations for a given pathogenic mutation from a BAM file of DNA sequencing data.
#' 
#' @param bam.file A character file name of the BAM file containing aligned reads to be processed.
#' @param out.dir A character file path to write output files.
#' @param reference A character variable specifying the reference genome version (hg19, hg38, mm10) or a FASTA file containing the open reading frames of reference sequences.
#' @param pathog.mut A character variable specifying the genomic position of pathogenic mutation following the HGVS-like syntax for substitution, deletion, insertion, deletion-insertion (delins), or duplication.
#' @param gene.name A character gene name for the pathogenic mutation.
#' @param transcript.id A character Ensembl Transcript ID for the pathogenic mutation.
#' @param detection.window A non-negative integer specifying the length of flanking regions to be added to both ends of pathogenic mutation locus for detecting reversion mutations. Default is 100.
#' @param splice.region A positive integer specifying the length of splicing junction region to be considered in introns. Default is 8.
#' @param check.soft.clipping A logical value indicating whether soft-clipped reads to be realigned. Default is TRUE.
#' @param softClippedReads.realign.window A non-negative integer specifying the length of flanking regions to be added to both ends of pathogenic mutation locus for realigning soft-clipped reads. Default is 1000.
#' @param softClippedReads.realign.match A non-negative integer specifying the scoring for a nucleotide match for realigning soft-clipped reads. Default is 1.
#' @param softClippedReads.realign.mismatch A non-negative integer specifying the scoring for a nucleotide mismatch for realigning soft-clipped reads. Default is 4.
#' @param softClippedReads.realign.gapOpening A non-negative integer specifying the cost for opening a gap in the realignment of soft-clipped reads. Default is 6.
#' @param softClippedReads.realign.gapExtension A non-negative integer specifying the incremental cost incurred along the length of the gap in the realignment of soft-clipped reads. Default is 0.
#' @param check.wildtype.reads A logical value indicating whether wild type reads to be processed as revertant-to-wildtype reads. Default is FALSE.
#' @param is.paired.end A logical value indicating whether reads in BAM file are paired-end (TRUE) or single-end (FALSE). Default is TRUE.
#' @param keep.duplicate.reads A logical value indicating whether duplicated reads in the BAM file to be processed (TRUE) or discarded (FALSE). Default is TRUE.
#' @param keep.secondary.alignment A logical value indicating whether secondary alignment reads in the BAM file to be processed (TRUE) or discarded (FALSE). Default is TRUE.
#' @param keep.supplementary.alignment A logical value indicating whether supplementary alignment reads in the BAM file to be processed (TRUE) or discarded (FALSE). Default is TRUE.
#' @param minimum.mapping.quality A non-negative integer specifying the minimum mapping quality of reads in the BAM file to be processed. Default is 0. 
#' @param verbose A logical value indicating whether progress logging to be printed to stdout. Default is TRUE.
#' @param out.failed.reads A logical value indicating whether the name of failed reads to be written to the '.failed_reads.txt' file. Default is FALSE.
#' 
#' @return Results written into output directory:  
#' \itemize{
#' \item ".reversions.txt" contains all reversions identified for the pathogenic mutation from the BAM file.  
#' \item ".split_mutations.txt" contains information of each single mutation in a reversion.  
#' \item ".revert_assembly.bam" contains all reads realigned to the pathogenic mutation.  
#' \item ".revert_assembly.bam.bai" is the index file for '.revert_assembly.bam'.  
#' \item ".revert_settings.txt" contains the summary of running parameters and processed reads.  
#' \item ".failed_reads.txt" (optional) contains the names of reads failed for reversion detection.  
#' } 
#' For more details of the output files see \href{../doc/help.html}{the help vignette}
#' 
#' @importFrom Rsamtools BamFile
#' @importFrom Rsamtools ScanBamParam
#' @importFrom Rsamtools scanBamHeader
#' @importFrom Rsamtools scanBamFlag
#' @importFrom Rsamtools scanBamWhat
#' @importFrom Rsamtools scanBam
#' @importFrom Rsamtools asBam
#' @importFrom BSgenome getBSgenome
#' @importFrom BSgenome getSeq
#' @importFrom Biostrings readDNAStringSet
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#' @importFrom stats quantile
#' @importFrom utils installed.packages
#' @importFrom utils write.table
#' 
#' @examples
#' \donttest{
#' getReversions( 
#'     bam.file = system.file("extdata", "toy_data_1.bam", package="revert"), 
#'     out.dir = tempdir(), 
#'     reference = "hg19", 
#'     pathog.mut = "chr13:g.32913319_32913320delTG", 
#'     gene.name = "BRCA2", 
#'     transcript.id = "ENST00000544455")
#' } 
#' # For more examples see the help vignette
#' 
#' @export getReversions
getReversions <- function(
		bam.file, 
		out.dir, 
		reference, 
		pathog.mut, 
		gene.name = NULL, 
		transcript.id = NULL, 
		detection.window = 100, 
		splice.region = 8, 
		check.soft.clipping = TRUE, 
		softClippedReads.realign.window = 1000, 
		softClippedReads.realign.match = 1, 
		softClippedReads.realign.mismatch = 4, 
		softClippedReads.realign.gapOpening = 6, 
		softClippedReads.realign.gapExtension = 0, 
		check.wildtype.reads = FALSE, 
		is.paired.end = TRUE, 
		keep.duplicate.reads = TRUE, 
		keep.secondary.alignment = TRUE, 
		keep.supplementary.alignment = TRUE, 
		minimum.mapping.quality = 0, 
		verbose = TRUE, 
		out.failed.reads = FALSE
	){
	
	#######################
	# parameter check and preprocess
	#######################
    time.start <- proc.time()
    
    ### verbose
    if(verbose){
        message("##################################################")
        message("Summary of initial parameters")
        message("##################################################")
        message("bam.file: ", bam.file)
        message("out.dir: ", out.dir)
        message("reference: ", reference)
        message("pathog.mut: ", pathog.mut)
        message("gene.name: ", gene.name)
        message("transcript.id: ", transcript.id)
        message("detection.window: ", detection.window)
        message("splice.region: ", splice.region)
        message("check.soft.clipping: ", check.soft.clipping)
        message("softClippedReads.realign.window: ", softClippedReads.realign.window)
        message("softClippedReads.realign.match: ", softClippedReads.realign.match)
        message("softClippedReads.realign.mismatch: ", softClippedReads.realign.mismatch)
        message("softClippedReads.realign.gapOpening: ", softClippedReads.realign.gapOpening)
        message("softClippedReads.realign.gapExtension: ", softClippedReads.realign.gapExtension)
        message("check.wildtype.reads: ", check.wildtype.reads)
        message("is.paired.end: ", is.paired.end)
        message("keep.duplicate.reads: ", keep.duplicate.reads)
        message("keep.secondary.alignment: ", keep.secondary.alignment)
        message("keep.supplementary.alignment: ", keep.supplementary.alignment)
        message("minimum.mapping.quality: ", minimum.mapping.quality)
        message("verbose: ", verbose)
        message("out.failed.reads: ", out.failed.reads)
        message("##################################################")
        message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Initializing parameter settings")
    }
    
    if(!dir.exists(out.dir)){
        dir.create(out.dir, recursive = TRUE)
    }
    
    if (!file.exists(bam.file)) {
        stop("BAM file was not found at ", bam.file)
    }else{
        bamfile <- Rsamtools::BamFile(bam.file)
        bam.header <- Rsamtools::scanBamHeader(bamfile)
        bam.header.chr <- names(bam.header$targets)
        bam.header.text <- bam.header$text
    }
	
    ### check ref pkg
    if( reference %in% c("hg19","hg38","mm10") ){
        
        instl.pkgs <- utils::installed.packages()[,1]
        
        if( reference == "hg19" ){
            reference.pkgs <- c("BSgenome.Hsapiens.UCSC.hg19","EnsDb.Hsapiens.v75")
        }else if( reference == "hg38" ){
            reference.pkgs <- c("BSgenome.Hsapiens.UCSC.hg38","EnsDb.Hsapiens.v86")
        }else if( reference == "mm10" ){
            reference.pkgs <- c("BSgenome.Mmusculus.UCSC.mm10", "EnsDb.Mmusculus.v79")
        }
        
        is.ref.pkgs.avail <- reference.pkgs %in% instl.pkgs
        
        if( !is.ref.pkgs.avail[1] & !is.ref.pkgs.avail[2] ){
            return(message("ACTION: Please install ", reference, " reference packages ", reference.pkgs[1], " and ", reference.pkgs[2]))
        }else if( !is.ref.pkgs.avail[1] & is.ref.pkgs.avail[2] ){
            return(message("ACTION: Please install ", reference, " reference package ", reference.pkgs[1]))
        }else if( is.ref.pkgs.avail[1] & !is.ref.pkgs.avail[2] ){
            return(message("ACTION: Please install ", reference, " reference package ", reference.pkgs[2]))
        }
        
        reference.genome <- BSgenome::getBSgenome(reference.pkgs[1])
        ensdb.envir <- loadNamespace(reference.pkgs[2])
        ensdb <- get(reference.pkgs[2], envir=ensdb.envir, inherits=FALSE)
        
    }else if( grepl( ".fa$|.fasta$", reference ) ){
        
        if (!file.exists(reference)) {
            stop("Reference FASTA file was not found at ", reference)
        }
        reference.genome <- Biostrings::readDNAStringSet(reference, format="fasta", nrec=-1L, skip=0L, seek.first.rec=FALSE, use.names=TRUE)
        ensdb <- NA
        
    }else{
        return(message("Reference ", reference, " is not supported.\nThe package currently supports the hg19, hg38, mm10 genome assemblies, or a customised FASTA file (.fa or .fasta) containing open read frames of the reference DNA sequence."))
    }
    
    ### process pothog mut
    pathog.mut.obj <- processPathogMut(pathog.mut, reference.genome)
    chr.id <- pathog.mut.obj$chr.id
    chr.name <- pathog.mut.obj$chr.name
    chr.length <- length( unlist( BSgenome::getSeq(reference.genome, chr.name) ) )
    
    ### check gene/tx for hg19/hg38/mm10
    ref.genome.class <- class(reference.genome)
    
    if( ref.genome.class=="BSgenome" ){
        
        if( is.null(gene.name) & is.null(transcript.id) ){
            return(message("ACTION: Gene name and transcript Ensembl ID of pathogenic mutation are required"))
        }else if( is.null(gene.name) & !is.null(transcript.id) ){
            return(message("ACTION: Gene name of pathogenic mutation is required"))
        }else if( !is.null(gene.name) & is.null(transcript.id) ){
            return(message("ACTION: Transcript Ensemnl ID of pathogenic mutation is required"))
        }
        
        gene.name <- as.character(gene.name)
        if(reference=="hg19" | reference=="hg38"){
            gene.name <- toupper(gene.name)
        }else if(reference=="mm10"){
            gene.name <- paste0(toupper(substring(tolower(gene.name), 1, 1)), substring(tolower(gene.name), 2))
        }
        
        gene.obj <- getGeneInfo(gene.name, chr.id, ensdb)
        if( is.na( gene.obj$gene.id ) ){
            return(message("No protein-coding genes were found in chromosome ", chr.id, " with gene name ", gene.name, " in Ensembl annotation ", reference.pkgs[2]))
        }
        
        tx.id <- as.character(transcript.id)
        if( ! (tx.id %in% gene.obj$tx.list) ){
            return(message("Transcript ", tx.id, " was not mapped to gene ", gene.name, " in Ensembl annotation ", reference.pkgs[2]))
        }
        
        pathog.mut.obj$strand <- gene.obj$strand
        pathog.mut.obj$gene.name <- gene.name
        pathog.mut.obj$gene.id <- gene.obj$gene.id
        pathog.mut.obj$tx.id <- tx.id
        
        pathog.mut.cds.anno <- annoPathogMutToCds(pathog.mut.obj, ensdb, splice.region)
        
        if(pathog.mut.cds.anno$mut.cds.location=="out_of_cds"){
            return(message("Pathogenic mutation is an intergenic or UTR variant which is not supported by the package."))
        }else if(pathog.mut.cds.anno$mut.cds.location=="intronic"){
            return(message("Pathogenic mutation is a non-splicing intronic variant which is not supported by the package."))
        }else if(pathog.mut.cds.anno$mut.cds.location=="LGR"){
            return(message("Pathogenic mutation is a large genomic rearrangement which is not supported by the package."))
        }
        
        pathog.mut.obj$mut.cds.location <- pathog.mut.cds.anno$mut.cds.location
        pathog.mut.obj$mut.start.to.cds <- pathog.mut.cds.anno$mut.start.to.cds
        pathog.mut.obj$mut.end.to.cds <- pathog.mut.cds.anno$mut.end.to.cds
        pathog.mut.obj$exon.id <- pathog.mut.cds.anno$mut.exon.id
        
        if( ! (chr.id%in%bam.header.chr | chr.name%in%bam.header.chr) ){
            return(message("BAM file does not contain the chromosome identifier ", chr.id, " or ", chr.name, "."))
        }else if( chr.id%in%bam.header.chr ){
            chr.bam <- chr.id
        }else if( chr.name%in%bam.header.chr ){
            chr.bam <- chr.name
        }
        
    }else if( ref.genome.class=="DNAStringSet" ){
        
        gene.obj <- NA
        tx.id <- NA
        pathog.mut.obj$strand <- "+"
        pathog.mut.obj$gene.name <- NA
        pathog.mut.obj$gene.id <- NA
        pathog.mut.obj$tx.id <- NA
        pathog.mut.obj$mut.cds.location <- "exonic"
        pathog.mut.obj$mut.start.to.cds <- NA
        pathog.mut.obj$mut.end.to.cds <- NA
        pathog.mut.obj$exon.id <- NA
        
        if( ! (chr.name%in%bam.header.chr) ){
            return(message("BAM file does not contain the reference sequence identifier ", chr.name, "."))
        }else{
            chr.bam <- chr.name
        }
        
    }
    
    pathog.mut.obj$chr.bam <- chr.bam
    
    ### check detection windows
    if( detection.window<0 ){
        stop("The detection.window must be an nonnegative integer. Defalut is 100.")
    }
    detection.window.left <- detection.window
    detection.window.right <- detection.window
    
    roi.start <- pathog.mut.obj$mut.start - detection.window.left
    roi.end <- pathog.mut.obj$mut.end + detection.window.right
    if( roi.start < 1 ){
        roi.start <- 1
        detection.window.left <- pathog.mut.obj$mut.start - roi.start
    }
    if( roi.end > chr.length ){
        roi.end <- chr.length
        detection.window.right <- roi.end - pathog.mut.obj$mut.end
    }
    
    ### check realign windows
    if( softClippedReads.realign.window<0 ){
        stop("The softClippedReads.realign.window must be an nonnegative integer. Defalut is 1000 when check.soft.clipping is TRUE.")
    }
    if( !check.soft.clipping ){
        softClippedReads.realign.window <- 0
    }
    
    softClippedReads.realign.window.left <- softClippedReads.realign.window
    softClippedReads.realign.window.right <- softClippedReads.realign.window
    
    if( check.soft.clipping & grepl("^splice", pathog.mut.obj$mut.cds.location) ){
        realn.soft.clipping.left <- cdsToGenome(pathog.mut.obj$mut.start.to.cds, tx.id, ensdb) - softClippedReads.realign.window.left
        realn.soft.clipping.right <- cdsToGenome(pathog.mut.obj$mut.end.to.cds, tx.id, ensdb) + softClippedReads.realign.window.right
        if( realn.soft.clipping.left < 1 ){
            realn.soft.clipping.left <- 1
        }
        if( realn.soft.clipping.right > chr.length ){
            realn.soft.clipping.right <- chr.length
        }
        softClippedReads.realign.window.left <- pathog.mut.obj$mut.start - realn.soft.clipping.left
        softClippedReads.realign.window.right <- realn.soft.clipping.right - pathog.mut.obj$mut.end
    }
    
	#######################
	# MAIN
	#######################
	out.prefix <- sub(".bam$","",basename(bam.file))
	out.file.sam <- file.path(out.dir, paste0(out.prefix, ".revert_temp.sam") )
	out.file.rev.list <- file.path(out.dir, paste0(out.prefix, ".reversions.txt") )
	out.file.rev.mutation <- file.path(out.dir, paste0(out.prefix, ".split_mutations.txt") )
	out.file.revert.settings <- file.path(out.dir, paste0(out.prefix, ".revert_settings.txt") )
	out.file.failed.reads <- file.path(out.dir, paste0(out.prefix, ".failed_reads.txt") )
    
	file.create(out.file.revert.settings)
	file.create(out.file.sam)
	if(out.failed.reads){
	    file.create(out.file.failed.reads)
	}
	
    ### to setting file
	cat("##################################################\n", file=out.file.revert.settings, append=TRUE)
	cat("Summary of running parameters\n", file=out.file.revert.settings, append=TRUE)
	cat("##################################################\n", file=out.file.revert.settings, append=TRUE)
	cat("bam.file:", bam.file, "\n", file=out.file.revert.settings, append=TRUE)
	cat("out.dir:", out.dir,"\n", file=out.file.revert.settings, append=TRUE)
	cat("reference:", reference,"\n", file=out.file.revert.settings, append=TRUE)
	cat("pathog.mut:", pathog.mut,"\n", file=out.file.revert.settings, append=TRUE)
	cat("gene.name:", gene.name,"\n", file=out.file.revert.settings, append=TRUE)
	cat("transcript.id:", transcript.id,"\n", file=out.file.revert.settings, append=TRUE)
	cat("detection.window:", detection.window,"\n", file=out.file.revert.settings, append=TRUE)
	cat("splice.region:", splice.region,"\n", file=out.file.revert.settings, append=TRUE)
	cat("check.soft.clipping:", check.soft.clipping,"\n", file=out.file.revert.settings, append=TRUE)
	cat("softClippedReads.realign.window:", softClippedReads.realign.window,"\n", file=out.file.revert.settings, append=TRUE)
	cat("softClippedReads.realign.match:", softClippedReads.realign.match,"\n", file=out.file.revert.settings, append=TRUE)
	cat("softClippedReads.realign.mismatch:", softClippedReads.realign.mismatch,"\n", file=out.file.revert.settings, append=TRUE)
	cat("softClippedReads.realign.gapOpening:", softClippedReads.realign.gapOpening,"\n", file=out.file.revert.settings, append=TRUE)
	cat("softClippedReads.realign.gapExtension:", softClippedReads.realign.gapExtension,"\n", file=out.file.revert.settings, append=TRUE)
	cat("check.wildtype.reads:", check.wildtype.reads,"\n", file=out.file.revert.settings, append=TRUE)
	cat("is.paired.end:", is.paired.end,"\n", file=out.file.revert.settings, append=TRUE)
	cat("keep.duplicate.reads:", keep.duplicate.reads,"\n", file=out.file.revert.settings, append=TRUE)
	cat("keep.secondary.alignment:", keep.secondary.alignment,"\n", file=out.file.revert.settings, append=TRUE)
	cat("keep.supplementary.alignment:", keep.supplementary.alignment,"\n", file=out.file.revert.settings, append=TRUE)
	cat("minimum.mapping.quality:", minimum.mapping.quality,"\n", file=out.file.revert.settings, append=TRUE)
	cat("verbose:", verbose,"\n", file=out.file.revert.settings, append=TRUE)
	cat("out.failed.reads:", out.failed.reads,"\n", file=out.file.revert.settings, append=TRUE)

	### verbose
	if(verbose){
	    message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Extracting reads from BAM file ", bam.file)
	}
	
	bam.rg.tag <- bam.header.text[ names(bam.header.text)=="@RG" ]
	if( length(bam.rg.tag)==0 ){
	    out.rg.tag.list <- list(
	        c("ID:Revertant", "CN:Unknown", "PL:Unknown", "PU:Unknown", paste0("SM:",out.prefix) ), 
	        c("ID:NonRevertant", "CN:Unknown", "PL:Unknown", "PU:Unknown", paste0("SM:",out.prefix) ) 
	    )
	}else{
	    rg.tag <- bam.rg.tag[[1]]
	    out.rg.tag1 <- rg.tag
	    out.rg.tag1[1] <- "ID:Revertant"
	    out.rg.tag2 <- rg.tag
	    out.rg.tag2[1] <- "ID:NonRevertant"
	    out.rg.tag.list <- list( out.rg.tag1, out.rg.tag2 )
	}
	names(out.rg.tag.list) <- c("@RG", "@RG")
	
	out.bam.header.text <- c(bam.header.text[names(bam.header.text)!="@RG"], out.rg.tag.list)
	
	for(i in 1:length(out.bam.header.text)){
		out.line <- NULL
		out.line <- paste(c(names(out.bam.header.text)[i], out.bam.header.text[[i]]), sep="\t")
		cat(out.line, file=out.file.sam, sep="\t", append=TRUE)
		cat("\n", file=out.file.sam, append=TRUE)
	}
	
	aln.gr <- GenomicRanges::GRanges(
	    seqnames = chr.bam, 
	    ranges = IRanges::IRanges(
	        start = max(pathog.mut.obj$mut.start - softClippedReads.realign.window.left, 1), 
	        end = min(pathog.mut.obj$mut.end + softClippedReads.realign.window.right, chr.length)
        )
    )
	
	is.paired <- ifelse(is.paired.end, TRUE, NA)
	has.unmapped.mate <- ifelse(is.paired.end, FALSE, NA)
	is.duplicate.reads <- ifelse(keep.duplicate.reads, NA, FALSE)
	is.secondary.alignment <- ifelse(keep.secondary.alignment, NA, FALSE)
	is.supplementary.alignment <- ifelse(keep.supplementary.alignment, NA, FALSE)
	
	params <- Rsamtools::ScanBamParam(
		which = aln.gr, 
		flag = Rsamtools::scanBamFlag(
		    isPaired = is.paired, 
		    hasUnmappedMate = has.unmapped.mate, 
			isUnmappedQuery = FALSE, 
			isNotPassingQualityControls = FALSE, 
			isDuplicate = is.duplicate.reads, 
			isSecondaryAlignment = is.secondary.alignment, 
			isSupplementaryAlignment = is.supplementary.alignment 
		),
		what = Rsamtools::scanBamWhat(),
		mapqFilter = minimum.mapping.quality
	)
	
	aln <- Rsamtools::scanBam(bamfile, param = params)[[1]]

	DP.done <- 0
	DP.failed <- 0
	DP.total <- 0
	DP.wt.total <- 0
	DP.mt.total <- 0
	DP.rpm.total <- 0
	DP.rev.total <- 0
	DP.mt.rev <- 0
	DP.rpm.rev <- 0
	DP.wt.rev <- 0
	DP.rev.uniq <- 0
	
	rev.list.full <- c()
	rev.mut.tbl <- c()

	if( length(aln$pos) > 0 ){
		
	    ### verbose
	    if(verbose){
            message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Processing ", length(aln$pos), " reads")
	    }
	    
	    progress.flag <- floor( stats::quantile(1:length(aln$pos), seq(0.01, 1, 0.01)) )
	    
		for(i in 1:length(aln$pos)){
            
			reads.obj <- data.frame(
				QNAME = aln$qname[i],
				FLAG = aln$flag[i],
				RNAME = aln$rname[i],
				POS = aln$pos[i],
				MAPQ = aln$mapq[i],
				CIGAR = aln$cigar[i],
				RNEXT = aln$mrnm[i],
				PNEXT = aln$mpos[i], 
				TLEN = aln$isize[i],
				SEQ = as.character(aln$seq[i]),
				QUAL = as.character(aln$qual[i]),
				stringsAsFactors = FALSE
			)
			
			rev.call <- tryCatch(
			    callSingleRead(
    				reference.genome,
    				ensdb, 
    				reads.obj,
    				pathog.mut.obj,
    				roi.start,
    				roi.end,
    				check.soft.clipping,
    				softClippedReads.realign.match,
    				softClippedReads.realign.mismatch,
    				softClippedReads.realign.gapOpening,
    				softClippedReads.realign.gapExtension,
    				softClippedReads.realign.window.left,
    				softClippedReads.realign.window.right,
    				check.wildtype.reads
    			),
			    error = function(e){
			        list(
			            realigned.reads.obj = reads.obj,
			            flags = list(
			                is.reads.failed = TRUE,
			                is.reads.target = FALSE,
			                is.reads.wt = FALSE,
			                is.reads.mt = FALSE,
			                is.reads.rpm = FALSE,
			                is.reversion = FALSE
			            ),
			            rev.event = NA,
			            rev.tbl = NA
			        )
			    } 
			)
			
			reads.obj.out <- rev.call$realigned.reads.obj
			is.reads.failed <- rev.call$flags$is.reads.failed
			is.reads.target <- rev.call$flags$is.reads.target
			is.reads.wt <- rev.call$flags$is.reads.wt
			is.reads.mt <- rev.call$flags$is.reads.mt
			is.reads.rpm <- rev.call$flags$is.reads.rpm
			is.reversion <- rev.call$flags$is.reversion
			rev.event <- rev.call$rev.event
			rev.tbl <- rev.call$rev.tbl
			
			if(is.reads.failed){
			    DP.failed <- DP.failed + 1
			    if(out.failed.reads){
			        cat(reads.obj.out$QNAME, "\n", file=out.file.failed.reads, append=TRUE)
			    }
			}else{
			    DP.done <- DP.done + 1
			}
			
			if(is.reads.target){
			    
			    DP.total <- DP.total + 1
			    if(is.reads.wt){
			        DP.wt.total <- DP.wt.total + 1
			    }
			    if(is.reads.mt){
			        DP.mt.total <- DP.mt.total + 1
			    }
			    if(is.reads.rpm){
			        DP.rpm.total <- DP.rpm.total + 1
			    }
			    
			    if(is.reversion){
			        reads.obj.out$RG <- "RG:Z:Revertant"
			    }else{
			        reads.obj.out$RG <- "RG:Z:NonRevertant"
			    }
			    write.table(reads.obj.out, file=out.file.sam, sep="\t", quote=FALSE, row.names=FALSE, col.names=FALSE, append=TRUE)
			}
			
			if(is.reversion){

				DP.rev.total <- DP.rev.total + 1
				if(is.reads.mt){
					DP.mt.rev <- DP.mt.rev + 1
				}
				if(is.reads.rpm){
					DP.rpm.rev <- DP.rpm.rev + 1
				}
				if(is.reads.wt & check.wildtype.reads){
					DP.wt.rev <- DP.wt.rev + 1
				}
			}
			
			is.new.reversion <- FALSE
			
			if( is.reversion ){
				if( is.null(rev.list.full) ){
					is.new.reversion <- TRUE
					DP.rev.uniq <- DP.rev.uniq + 1
					rev.list.full <- data.frame(
						rev_id = paste0( "rev", DP.rev.uniq ),
						rev_freq = 1,
						rev_event = rev.event,
						stringsAsFactors = FALSE 
					)
				}else{
					if( rev.event %in% rev.list.full$rev_event ){
						rev.list.full$rev_freq[ rev.list.full$rev_event==rev.event ] <- rev.list.full$rev_freq[ rev.list.full$rev_event==rev.event ] + 1  
					}else{
						is.new.reversion <- TRUE
						DP.rev.uniq <- DP.rev.uniq + 1
						rev.list.full[ nrow(rev.list.full)+1 , ] <- list( paste0("rev",DP.rev.uniq), 1, rev.event)
					}
				}
				
				if( is.new.reversion ){
					rev.tbl$rev_id <- paste0( "rev", rep(DP.rev.uniq, nrow(rev.tbl) ) )
					rev.tbl$mut_id <- paste(rev.tbl$rev_id, rev.tbl$rev_mut_number, sep=".")
					rev.mut.tbl <- rbind(rev.mut.tbl, rev.tbl)
				}
			}
		    
			### verbose
			if( verbose & i%in%progress.flag ){
			    message(format(Sys.time(),"%H:%M:%OS3"), " INFO  ProgressMeter ==> ", i , " (", names(progress.flag)[max(which(progress.flag==i))], ") reads")
			}
			
		}
		
	    if(verbose){
	        message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Processed ", length(aln$pos), " reads: ", DP.done, " done, ", DP.failed, " failed." )
	    }
	    
	}
	
	if(verbose){
	    message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Writing outputs to ", out.dir)
	}
	
	### to setting file
	cat("\n", file=out.file.revert.settings, append=TRUE)
	cat("##################################################\n", file=out.file.revert.settings, append=TRUE)
	cat("Summary of evidence reads\n", file=out.file.revert.settings, append=TRUE)
	cat("##################################################\n", file=out.file.revert.settings, append=TRUE)
	cat("Total reads aligned to pathogenic mutation locus:", DP.total, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Reads with wildtype sequence at pathogenic mutation locus:", DP.wt.total, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Reads with pathogenic mutation:", DP.mt.total, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Reads with a different mutation at pathogenic mutation locus:", DP.rpm.total, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Total reads with reversions:", DP.rev.total, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Reads with reversions and pathogenic mutation retained:", DP.mt.rev, "\n", file=out.file.revert.settings, append=TRUE)
	cat("Reads with reversions but pathogenic mutation not-retained:", DP.rpm.rev, "\n", file=out.file.revert.settings, append=TRUE)
	if( check.wildtype.reads ){
		cat("Reads revertant to wildtype:", DP.wt.rev, "\n", file=out.file.revert.settings, append=TRUE)
	}
	cat("Unique reversions:", DP.rev.uniq, "\n", file=out.file.revert.settings, append=TRUE)
	
	rev.list.out.cols <- c(
		"pathogenic_mutation",
		"pathogenic_mutation_left_aligned",
		"reversion_id",
		"reversion_frequency",
		"pathogenic_mutation_retained",
		"reversion",
		"reads_total",
		"reads_wildtype",
		"reads_withPathogenicMutation",
		"reads_withReplacementMutation",
		"mutations_in_reversion"
	)
	
	rev.mut.out.cols <- c(
		"rev_id",
		"mut_id",
		"mut_type",
		"mut_hgvs",
		"mut_length_change",
		"pathog_mut_hgvs",
		"dist_to_pathog_mut" 
	)
	
	if( is.null(rev.list.full) ){
		rev.list.tbl <- data.frame( matrix(ncol=length(rev.list.out.cols), nrow=0 ) )
		colnames(rev.list.tbl) <- rev.list.out.cols
		rev.mut.tbl <- data.frame( matrix(ncol=length(rev.mut.out.cols), nrow=0) ) 
		colnames(rev.mut.tbl) <- rev.mut.out.cols
	}else{
		rev.list.tbl <- data.frame(
		    pathogenic_mutation = pathog.mut.obj$mut.raw,
		    pathogenic_mutation_left_aligned = ifelse( pathog.mut.obj$mut.type=="DEL"|pathog.mut.obj$mut.type=="INS", pathog.mut.obj$mut.hgvs, NA),
		    reversion_id = rev.list.full$rev_id,
		    reversion_frequency = rev.list.full$rev_freq,
		    pathogenic_mutation_retained = sapply(strsplit(rev.list.full$rev_event,"|",fixed=T), function(x)x[1] ),
			reversion = sapply(strsplit(rev.list.full$rev_event,"|",fixed=T), function(x)x[2] ),
			reads_total = DP.total,
			reads_wildtype = DP.wt.total, 
			reads_withPathogenicMutation = DP.mt.total, 
			reads_withReplacementMutation = DP.rpm.total,
			mutations_in_reversion = sapply(strsplit(rev.list.full$rev_event,";",fixed=T), function(x)length(x) ),
			stringsAsFactors = FALSE
		)
		rev.mut.tbl <- rev.mut.tbl[, rev.mut.out.cols]
	}
	colnames(rev.mut.tbl) <- c( "reversion_id","mutation_id","mutation_type","mutation","mutation_length_change","pathogenic_mutation","distance_to_pathogenic_mutation" )
	
	write.table(rev.list.tbl, out.file.rev.list, sep="\t", quote=FALSE, row.names=FALSE, col.names=TRUE)
	write.table(rev.mut.tbl, out.file.rev.mutation, sep="\t", quote=FALSE, row.names=FALSE, col.names=TRUE)
	
	Rsamtools::asBam(
	    file = out.file.sam, 
	    destination = paste0(out.dir,"/",out.prefix,".revert_assembly"), 
	    byQname = FALSE, 
	    overwrite = TRUE,
	    indexDestination = TRUE
	)
	
	file.remove(out.file.sam)
	
	time.end <- proc.time()
	time.elapsed <- round((time.end-time.start)["elapsed"]/60, 2)
	
	### verbose
	if(verbose){
	    message(format(Sys.time(),"%H:%M:%OS3"), " INFO  Reversion detection done. Elapsed time: ", time.elapsed, " minutes.")
	}
	
} 
		


