% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rtoqmd.R
\name{rtoqmd}
\alias{rtoqmd}
\title{Convert R Script to Quarto Markdown}
\usage{
rtoqmd(
  input_file,
  output_file = NULL,
  title = "My title",
  author = "Your name",
  format = "html",
  theme = NULL,
  render_html = TRUE,
  output_html_file = NULL,
  open_html = FALSE,
  code_fold = FALSE,
  number_sections = TRUE,
  lang = "en",
  show_source_lines = TRUE,
  use_styler = FALSE,
  use_lintr = FALSE,
  apply_styler = FALSE
)
}
\arguments{
\item{input_file}{Path to the input R script file}

\item{output_file}{Path to the output Quarto markdown file (optional, defaults to same name with .qmd extension)}

\item{title}{Title for the Quarto document (default: "My title"). Can be overridden by \code{# Title :} or \code{# Titre :} in the script}

\item{author}{Author name (default: "Your name"). Can be overridden by \code{# Author :} or \code{# Auteur :} in the script}

\item{format}{Output format - always "html" (parameter kept for backward compatibility)}

\item{theme}{Quarto theme for HTML output (default: NULL uses Quarto's default). See \url{https://quarto.org/docs/output-formats/html-themes.html} for available themes (e.g., "cosmo", "flatly", "darkly", "solar", "united")}

\item{render_html}{Logical, whether to render the .qmd file to HTML after creation (default: TRUE)}

\item{output_html_file}{Path to the output HTML file (optional, defaults to same name as .qmd file with .html extension)}

\item{open_html}{Logical, whether to open the HTML file in browser after rendering (default: FALSE, only used if render_html = TRUE)}

\item{code_fold}{Logical, whether to fold code blocks in HTML output (default: FALSE)}

\item{number_sections}{Logical, whether to number sections automatically in the output (default: TRUE)}

\item{lang}{Language for interface elements like table of contents title - "en" or "fr" (default: "en")}

\item{show_source_lines}{Logical, whether to add comments indicating original line numbers from the source R script at the beginning of each code chunk (default: TRUE). This helps maintain traceability between the documentation and the source code.}

\item{use_styler}{Logical, whether to apply styler code formatting and show differences in tabsets (default: FALSE). Requires the styler package to be installed.}

\item{use_lintr}{Logical, whether to run lintr code quality checks and display issues in tabsets (default: FALSE). Requires the lintr package to be installed.}

\item{apply_styler}{Logical, whether to apply styler formatting directly to the source R script file (default: FALSE). If TRUE, the input file will be modified with styled code. Requires use_styler = TRUE to take effect.}
}
\value{
Invisibly returns NULL. Creates a .qmd file and optionally renders it to HTML.
}
\description{
This function converts an R script to Quarto markdown format (.qmd), enabling you to leverage
all modern Quarto features. Unlike \code{knitr::spin()} which generates R Markdown (.Rmd),
\code{rtoqmd()} creates Quarto documents with access to advanced publishing capabilities,
modern themes, native callouts, Mermaid diagrams, and the full Quarto ecosystem.
}
\details{
It recognizes RStudio code sections with different levels:
- ## Title #### creates a level 2 header
- ### Title ==== creates a level 3 header
- #### Title ---- creates a level 4 header
Regular comments are converted to plain text.
Code blocks are wrapped in standard R code chunks. The YAML header includes
\code{execute: eval: false} and \code{execute: echo: true} options for static
documentation purposes, and \code{embed-resources: true} to create self-contained
HTML files. See \url{https://quarto.org/docs/output-formats/html-basics.html#self-contained}.
}
\section{Metadata Detection}{

The function automatically extracts metadata from special comment lines in your R script:
\itemize{
  \item \strong{Title}: Use \code{# Title : Your Title} or \code{# Titre : Votre Titre}
  \item \strong{Author}: Use \code{# Author : Your Name} or \code{# Auteur : Votre Nom}
  \item \strong{Date}: Use \code{# Date : YYYY-MM-DD}
  \item \strong{Description}: Use \code{# Description : Your description} (also accepts \code{# Purpose} or \code{# Objectif})
}
If metadata is found in the script, it will override the corresponding function parameters.
These metadata lines are removed from the document body and only appear in the YAML header.

The Description field supports multi-line content. Continuation lines should start with \code{#}
followed by spaces and the text. The description ends at an empty line or a line without \code{#}.
}

\section{Hidden Comments}{

Comments that start with \code{#} immediately followed by a non-space character (e.g., \code{#NOTE:}, \code{#TODO:}, \code{#DEBUG})
are completely ignored during conversion and will not appear in the Quarto output.
This allows you to include private notes, debugging comments, or development annotations in your R scripts
that won't be visible in the rendered documentation.

Only comments with a space after \code{#} (e.g., \code{# This is a comment}) are converted to text in the output.
}

\section{Callouts}{

The function converts special comment patterns into Quarto callouts.
Callouts are special blocks that highlight important information.
Supported callout types: \code{note}, \code{tip}, \code{warning}, \code{caution}, \code{important}.

Syntax:
\itemize{
  \item \strong{With title}: \code{# callout-tip - Your Title}
  \item \strong{Without title}: \code{# callout-tip}
}

All subsequent comment lines become the callout content until an empty line or code is encountered.

Example in R script:
\preformatted{
# callout-note - Important Note
# This is the content of the note.
# It can span multiple lines.

x <- 1
}

Becomes in Quarto:
\preformatted{
::: {.callout-note title="Important Note"}
This is the content of the note.
It can span multiple lines.
:::
}
}

\section{Mermaid Diagrams}{

The function supports Mermaid diagrams for flowcharts, sequence diagrams, and visualizations.
Mermaid chunks start with a special comment, followed by options and diagram content.
Options use hash-pipe syntax and are converted to percent-pipe in the Quarto output.
Diagram content should not start with hash symbols. The chunk ends at a blank line or comment.
Supported types: flowchart, sequence, class, state, etc. See example file in inst/examples/example_mermaid.R.
}

\section{Tabsets}{

Create tabbed content panels for interactive navigation between related content.
Use hash tabset to start a tabset container, then define individual tabs with hash tab - Title.
Each tab can contain text, code, and other content. The tabset closes automatically when a new section starts.
Example: hash tabset, hash tab - Plot A, code or text content, hash tab - Plot B, more content.
}

\section{Roxygen2 Documentation}{

The function automatically detects and formats roxygen2 documentation blocks (starting with \code{#'})
into structured callouts that resemble pkgdown reference pages. The formatted documentation includes:
\itemize{
  \item \strong{Title}: Extracted from \code{@title} tag or first roxygen comment line
  \item \strong{Description}: From \code{@description} tag or initial paragraph
  \item \strong{Usage}: Function signature with parameters
  \item \strong{Arguments}: Each parameter from \code{@param} tags, formatted with parameter name in bold
  \item \strong{Value}: Return value description from \code{@return} tag
  \item \strong{Details}: Additional details from \code{@details} tag
  \item \strong{Examples}: Code examples from \code{@examples} tag, displayed in R code blocks
}
LaTeX-style formatting is automatically converted to Markdown:
\code{\\href\{url\}\{text\}} becomes \code{[text](url)},
\code{\\code\{text\}} becomes \code{`text`},
\code{\\strong\{text\}} becomes \code{**text**}, and
\code{\\emph\{text\}} becomes \code{*text*}.
Section headers within the callout use bold text instead of Markdown headers to avoid
interfering with the document's table of contents. See example file in inst/examples/example_roxygen.R.
}

\examples{
\donttest{
# Use example file included in package
example_file <- system.file("examples", "example.R", package = "quartify")

# Convert and render to HTML (output in temp directory)
output_qmd <- file.path(tempdir(), "output.qmd")
rtoqmd(example_file, output_qmd)

# Convert only, without rendering
rtoqmd(example_file, output_qmd, render_html = FALSE)

# Example with metadata in the R script:
# Create a script with metadata
script_with_metadata <- tempfile(fileext = ".R")
writeLines(c(
  "# Title : My Analysis",
  "# Author : Jane Doe", 
  "# Date : 2025-11-28",
  "# Description : Analyze iris dataset",
  "",
  "library(dplyr)",
  "iris \%>\% head()"
), script_with_metadata)

# Convert - metadata will override function parameters
output_meta <- file.path(tempdir(), "output_with_metadata.qmd")
rtoqmd(script_with_metadata, output_meta)

# Example with code quality checks (requires styler and lintr packages)
script_with_style_issues <- tempfile(fileext = ".R")
writeLines(c(
  "# Script with style issues",
  "",
  "x = 3  # Should use <- instead of =",
  "y <- 2",
  "",
  "z <- 10"
), script_with_style_issues)

# Convert with styler formatting
output_styled <- file.path(tempdir(), "output_styled.qmd")
rtoqmd(script_with_style_issues, output_styled, use_styler = TRUE)

# Convert with both styler and lintr
output_quality <- file.path(tempdir(), "output_quality.qmd")
rtoqmd(script_with_style_issues, output_quality, 
       use_styler = TRUE, use_lintr = TRUE)
}
}
