#' @title Get current columns in VecDyn datasets
#' @description Get all the current column names in VecDyn, alongside associated data if desired.
#' @author Francis Windram
#'
#' @param full whether to return all the data about current fields, or else just return the names
#' @param basereq an [httr2 request][httr2::request()] object, as generated by [vb_basereq()]. If `NA`, uses the default request.
#' @return A character vector (or dataframe) of column information.
#'
#' @examplesIf interactive()
#' find_vd_columns()
#'
#' @concept vecdyn
#'
#' @export
#'

find_vd_columns <- function(full = FALSE, basereq = vb_basereq()) {

  resplist <- tryCatch(
    {
      resp <- basereq |>
        req_url_path_append("vecdyn-columns") |>
        req_url_query("format" = "json") |>
        req_perform()
      list("resp" = resp, "err_code" = 0, "err_obj" = NULL)
    },
    error = function(e) {
      # Get the last response instead
      list("resp" = last_response(), "err_code" = 1, "err_obj" = e)
    }
  )

  if (resplist$err_code == 1) {
    cli::cli_abort("No records found.")
  }

  body <- resplist$resp |> resp_body_json()
  fields <- as.character(lapply(body, function(x) {
    return(x$field_name[[1]])
  }))
  if (full) {
    dataformat <- as.character(lapply(body, function(x) {
      return(x$data_format[[1]])
    }))
    definitions <- as.character(lapply(body, function(x) {
      return(x$definition[[1]])
    }))
    out <- data.frame(fields, dataformat, definitions)
    suppressMessages(colnames(out) <- c("Field", "Format", "Definition"))
    return(out)
  }

  return(fields)
}
