% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zzz.R
\docType{package}
\name{lessSEM}
\alias{lessSEM-package}
\alias{lessSEM}
\title{lessSEM}
\description{
\strong{Please see the vignettes and the readme on \href{https://jhorzek.github.io/lessSEM/}{GitHub}
for the most up to date description of the package}
}
\details{
\strong{lessSEM} (\strong{l}essSEM \strong{es}timates \strong{s}parse \strong{SEM}) is an R package for
regularized structural equation modeling (regularized SEM) with non-smooth
penalty functions (e.g., lasso) building on \href{https://github.com/yrosseel/lavaan}{\strong{lavaan}}.
\strong{lessSEM} is heavily inspired by the \href{https://github.com/Rjacobucci/regsem}{\strong{regsem}} package
and the \href{https://github.com/psyphh/lslx}{\strong{lslx}} packages that have similar functionality.
\strong{If you use lessSEM, please also cite \href{https://github.com/Rjacobucci/regsem}{regsem} and
and \href{https://github.com/psyphh/lslx}{lslx}!}

The objectives of \strong{lessSEM} are to provide ...
\enumerate{
\item a flexible framework for regularizing SEM and
\item optimizers for other SEM packages which can be used with an interface
similar to \code{optim}.
}

\strong{Important}: Please also check out the implementations of regularized SEM in the more
mature R packages \href{https://github.com/Rjacobucci/regsem}{\strong{regsem}} and \href{https://github.com/psyphh/lslx}{\strong{lslx}}.
Finally, you may want to check out the julia package
\href{https://github.com/StructuralEquationModels/StructuralEquationModels.jl}{\strong{StructuralEquationModels.jl}}.
\subsection{\href{https://github.com/Rjacobucci/regsem}{\strong{regsem}}, \href{https://github.com/psyphh/lslx}{\strong{lslx}}, and \strong{lessSEM}}{

The packages \href{https://github.com/Rjacobucci/regsem}{\strong{regsem}},
\href{https://github.com/psyphh/lslx}{\strong{lslx}}, and \strong{lessSEM} can all be used to
regularize basic SEM. In fact, as outlined above, \strong{lessSEM} is heavily inspired
by \href{https://github.com/Rjacobucci/regsem}{\strong{regsem}}
and \href{https://github.com/psyphh/lslx}{\strong{lslx}}. However, the packages differ in their targets: The objective
of \strong{lessSEM} is not to replace the more major packages \href{https://github.com/Rjacobucci/regsem}{\strong{regsem}}
and \href{https://github.com/psyphh/lslx}{\strong{lslx}}. Instead, our objective is to
provide method developers with a flexible framework for regularized SEM.
The following shows an incomplete comparison of some features implemented in
the three packages:\tabular{llll}{
    \tab \strong{regsem} \tab \strong{lslx} \tab \strong{lessSEM} \cr
   Model specification \tab based on lavaan \tab similar to lavaan \tab based on lavaan \cr
   Maximum likelihood estimation \tab Yes \tab Yes \tab Yes \cr
   Least squares estimation \tab No \tab Yes \tab No \cr
   Confidence Intervals \tab No \tab Yes \tab No \cr
   Missing Data \tab FIML \tab Auxiliary Variables \tab FIML \cr
   Multi-group models \tab No \tab Yes \tab Yes \cr
   Stability selection \tab Yes \tab No \tab No \cr
   Mixed penalties \tab No \tab No \tab Yes \cr
   Equality constraints \tab Yes \tab No \tab Yes \cr
   Parameter transformations \tab diff_lasso \tab No \tab Yes \cr
   Definition variables \tab No \tab No \tab Yes \cr
}


Because \strong{lessSEM} is fairly new, we currently recommend using \strong{lslx} for cases
that are covered by both, \strong{lessSEM} and \strong{lslx}.
}
}
\section{Introduction}{
You will find a short introduction to regularized SEM with the \strong{lessSEM}
package in \code{vignette('lessSEM', package = 'lessSEM')}. More information is also
provided in the documentation of the individual functions (e.g., see \code{?lessSEM::scad}).
Finally, you will find templates for a selection of models which can be used with \strong{lessSEM}
(e.g., the cross-lagged panel model) in the package \href{https://github.com/jhorzek/lessTemplates}{\strong{lessTemplates}}.
}

\section{Example}{
\if{html}{\out{<div class="sourceCode">}}\preformatted{library(lessSEM)
library(lavaan)

# Identical to regsem, lessSEM builds on the lavaan
# package for model specification. The first step
# therefore is to implement the model in lavaan.

dataset <- simulateExampleData()

lavaanSyntax <- "
      f =~ l1*y1 + l2*y2 + l3*y3 + l4*y4 + l5*y5 + 
           l6*y6 + l7*y7 + l8*y8 + l9*y9 + l10*y10 + 
           l11*y11 + l12*y12 + l13*y13 + l14*y14 + l15*y15
      f ~~ 1*f
      "

lavaanModel <- lavaan::sem(lavaanSyntax,
                           data = dataset,
                           meanstructure = TRUE,
                           std.lv = TRUE)

# Regularization:

lsem <- lasso(
  # pass the fitted lavaan model
  lavaanModel = lavaanModel,
  # names of the regularized parameters:
  regularized = c("l6", "l7", "l8", "l9", "l10",
                  "l11", "l12", "l13", "l14", "l15"),
  # in case of lasso and adaptive lasso, we can specify the number of lambda
  # values to use. lessSEM will automatically find lambda_max and fit
  # models for nLambda values between 0 and lambda_max. For the other
  # penalty functions, lambdas must be specified explicitly
  nLambdas = 50)

# use the plot-function to plot the regularized parameters:
plot(lsem)

# use the coef-function to show the estimates
coef(lsem)

# The best parameters can be extracted with:
coef(lsem, criterion = "AIC")
coef(lsem, criterion = "BIC")

# elements of lsem can be accessed with the @ operator:
lsem@parameters[1,]

# AIC and BIC for all tuning parameter configurations:
AIC(lsem)
BIC(lsem)

# cross-validation
cv <- cvLasso(lavaanModel = lavaanModel,
              regularized = c("l6", "l7", "l8", "l9", "l10",
                              "l11", "l12", "l13", "l14", "l15"),
              lambdas = seq(0,1,.1),
              standardize = TRUE)

# get best model according to cross-validation:
coef(cv)

#### Advanced ###
# Switching the optimizer # 
# Use the "method" argument to switch the optimizer. The control argument
# must also be changed to the corresponding function:
lsemIsta <- lasso(
  lavaanModel = lavaanModel,
  regularized = paste0("l", 6:15),
  nLambdas = 50,
  method = "ista",
  control = controlIsta(
    # Here, we can also specify that we want to use multiple cores:
    nCores = 2))

# Note: The results are basically identical:
lsemIsta@parameters - lsem@parameters
}\if{html}{\out{</div>}}
}

\section{Transformations}{
\strong{lessSEM} allows for parameter transformations which could, for instance, be used to test
measurement invariance in longitudinal models (e.g., Liang, 2018; Bauer et al., 2020).
A thorough introduction is provided in \code{vignette('Parameter-transformations', package = 'lessSEM')}.
As an example, we will test measurement invariance in the \code{PoliticalDemocracy}
data set.

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(lessSEM)
library(lavaan)
# we will use the PoliticalDemocracy from lavaan (see ?lavaan::sem)
model <- ' 
  # latent variable definitions
     ind60 =~ x1 + x2 + x3
     # assuming different loadings for different time points:
     dem60 =~ y1 + a1*y2 + b1*y3 + c1*y4
     dem65 =~ y5 + a2*y6 + b2*y7 + c2*y8

  # regressions
    dem60 ~ ind60
    dem65 ~ ind60 + dem60

  # residual correlations
    y1 ~~ y5
    y2 ~~ y4 + y6
    y3 ~~ y7
    y4 ~~ y8
    y6 ~~ y8
'

fit <- sem(model, data = PoliticalDemocracy)

# We will define a transformation which regularizes differences
# between loadings over time:

transformations <- "
// which parameters do we want to use?
parameters: a1, a2, b1, b2, c1, c2, delta_a2, delta_b2, delta_c2

// transformations:
a2 = a1 + delta_a2;
b2 = b1 + delta_b2;
c2 = c1 + delta_c2;
"

# setting delta_a2, delta_b2, or delta_c2 to zero implies measurement invariance
# for the respective parameters (a1, b1, c1)
lassoFit <- lasso(lavaanModel = fit, 
                  # we want to regularize the differences between the parameters
                  regularized = c("delta_a2", "delta_b2", "delta_c2"),
                  nLambdas = 100,
                  # Our model modification must make use of the modifyModel - function:
                  modifyModel = modifyModel(transformations = transformations)
)
}\if{html}{\out{</div>}}

Finally, we can extract the best parameters:

\if{html}{\out{<div class="sourceCode">}}\preformatted{coef(lassoFit, criterion = "BIC")
}\if{html}{\out{</div>}}

As all differences (\code{delta_a2}, \code{delta_b2}, and \code{delta_c2}) have been zeroed, we can
assume measurement invariance.
}

\section{Experimental Features}{
The following features are relatively new and you may still experience some bugs.
Please be aware of that when using these features.
\subsection{From \strong{lessSEM} to \strong{lavaan}}{

\strong{lessSEM} supports exporting specific models to \strong{lavaan}. This can be very useful when plotting the
final model. In our case, the best model is given by:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lambdaBest <- coef(lsem, criterion = "BIC")@tuningParameters$lambda 
}\if{html}{\out{</div>}}

We can get the \strong{lavaan} model with the parameters corresponding to those of the
regularized model with \code{lambda = lambdaBest} as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{lavaanModel <- lessSEM2Lavaan(regularizedSEM = lsem, 
                              lambda = lambdaBest)
}\if{html}{\out{</div>}}

The result can be plotted with, for instance, \href{https://github.com/SachaEpskamp/semPlot}{\strong{semPlot}}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(semPlot)
semPaths(lavaanModel,
         what = "est",
         fade = FALSE)
}\if{html}{\out{</div>}}
}

\subsection{Multi-Group Models and Definition Variables}{

\strong{lessSEM} supports multi-group SEM and, to some degree, definition variables.
Regularized multi-group SEM have been proposed by Huang (2018) and are
implemented in \strong{lslx} (Huang, 2020). Here, differences between groups are regularized.
A detailed introduction can be found in
\code{vignette(topic = "Definition-Variables-and-Multi-Group-SEM", package = "lessSEM")}.
Therein it is also explained how the multi-group SEM can be used to implement
definition variables (e.g., for latent growth curve models).
}

\subsection{Mixed Penalties}{

\strong{lessSEM} allows for defining different penalties for different parts
of the model. This feature is new and very experimental. Please keep that
in mind when using the procedure. A detailed introduction
can be found in \code{vignette(topic = "Mixed-Penalties", package = "lessSEM")}.

To provide a short example, we will regularize the loadings and the regression
parameters of the Political Democracy data set with different penalties. The
following script is adapted from \code{?lavaan::sem}.

\if{html}{\out{<div class="sourceCode">}}\preformatted{model <- ' 
  # latent variable definitions
     ind60 =~ x1 + x2 + x3 + c2*y2 + c3*y3 + c4*y4
     dem60 =~ y1 + y2 + y3 + y4
     dem65 =~ y5 + y6 + y7 + c*y8

  # regressions
    dem60 ~ r1*ind60
    dem65 ~ r2*ind60 + r3*dem60
'

lavaanModel <- sem(model,
                   data = PoliticalDemocracy)

# Let's add a lasso penalty on the cross-loadings c2 - c4 and 
# scad penalty on the regressions r1-r3
fitMp <- lavaanModel |>
  mixedPenalty() |>
  addLasso(regularized = c("c2", "c3", "c4"), 
           lambdas = seq(0,1,.1)) |>
  addScad(regularized = c("r1", "r2", "r3"), 
          lambdas = seq(0,1,.2),
          thetas = 3.7) |>
  fit()
}\if{html}{\out{</div>}}

The best model according to the BIC can be extracted with:

\if{html}{\out{<div class="sourceCode">}}\preformatted{coef(fitMp, criterion = "BIC")
}\if{html}{\out{</div>}}
}
}

\section{Optimizers}{
Currently, \strong{lessSEM} has the following optimizers:
\itemize{
\item (variants of) iterative shrinkage and thresholding (e.g., Beck & Teboulle, 2009;
Gong et al., 2013; Parikh & Boyd, 2013); optimization of cappedL1, lsp, scad, and mcp is based on Gong et al. (2013)
\item glmnet (Friedman et al., 2010; Yuan et al., 2012; Huang, 2020)
}

These optimizers are implemented based on the
\href{https://github.com/jhorzek/regCtsem}{\strong{regCtsem}} package. Most importantly,
\strong{all optimizers in lessSEM are available for other packages.}
There are three ways to implement them which are documented in
\code{vignette("General-Purpose-Optimization", package = "lessSEM")}.
In short, these are:
\enumerate{
\item using the R interface: All general purpose implementations of the functions
are called with prefix "gp" (\code{gpLasso}, \code{gpScad}, ...). More information and
examples can be found in the documentation of these functions (e.g., \code{?lessSEM::gpLasso},
\code{?lessSEM::gpAdaptiveLasso}, \code{?lessSEM::gpElasticNet}). The interface is similar to
the optim optimizers in R.
\item using Rcpp, we can pass C++ function pointers to the general purpose optimizers
\code{gpLassoCpp}, \code{gpScadCpp}, ... (e.g., \code{?lessSEM::gpLassoCpp})
\item All optimizers are implemented as C++ header-only files in \strong{lessSEM}. Thus,
they can be accessed from other packages using C++. The interface is similar
to that of the \href{https://ensmallen.org/}{\strong{ensmallen}} library. We have implemented
a simple example for elastic net regularization of linear regressions in the
\href{https://github.com/jhorzek/lessLM}{\strong{lessLM}} package. You can also find more
details on the general design of the optimizer interface in \code{vignette("The-optimizer-interface", package = "lessSEM")}.
}
}

\section{References}{
\subsection{R - Packages / Software}{
\itemize{
\item \href{https://github.com/yrosseel/lavaan}{lavaan} Rosseel, Y. (2012). lavaan: An R Package for Structural Equation Modeling. Journal of Statistical Software, 48(2), 1-36. https://doi.org/10.18637/jss.v048.i02
\item \href{https://github.com/Rjacobucci/regsem}{regsem}: Jacobucci, R. (2017). regsem:
Regularized Structural Equation Modeling. ArXiv:1703.08489. https://arxiv.org/abs/1703.08489
\item \href{https://github.com/psyphh/lslx}{lslx}: Huang, P.-H. (2020). lslx:
Semi-confirmatory structural equation modeling via penalized likelihood. Journal
of Statistical Software, 93(7). https://doi.org/10.18637/jss.v093.i07
\item \href{https://CRAN.R-project.org/package=fasta}{fasta}:
Another implementation of the fista algorithm (Beck & Teboulle, 2009).
\item \href{https://ensmallen.org/}{ensmallen}: Curtin, R. R., Edel, M., Prabhu, R. G.,
Basak, S., Lou, Z., & Sanderson, C. (2021). The ensmallen library for ﬂexible
numerical optimization. Journal of Machine Learning Research, 22, 1–6.
\item \href{https://github.com/jhorzek/regCtsem}{regCtsem}: Orzek, J. H., & Voelkle, M. C. (in press).
Regularized continuous time structural equation models: A network perspective. Psychological Methods.
}
}

\subsection{Regularized Structural Equation Modeling}{
\itemize{
\item Huang, P.-H., Chen, H., & Weng, L.-J. (2017). A Penalized Likelihood Method
for Structural Equation Modeling. Psychometrika, 82(2), 329–354. https://doi.org/10.1007/s11336-017-9566-9
\item Huang, P.-H. (2018). A penalized likelihood method for multi-group structural equation modelling. British Journal of Mathematical and Statistical Psychology, 71(3), 499–522. https://doi.org/10.1111/bmsp.12130
\item Jacobucci, R., Grimm, K. J., & McArdle, J. J. (2016). Regularized Structural
Equation Modeling. Structural Equation Modeling: A Multidisciplinary Journal, 23(4),
555–566. https://doi.org/10.1080/10705511.2016.1154793
}
}

\subsection{Penalty Functions}{
\itemize{
\item Candès, E. J., Wakin, M. B., & Boyd, S. P. (2008). Enhancing Sparsity by
Reweighted l1 Minimization. Journal of Fourier Analysis and Applications, 14(5–6),
877–905. https://doi.org/10.1007/s00041-008-9045-x
\item Fan, J., & Li, R. (2001). Variable selection via nonconcave penalized
likelihood and its oracle properties. Journal of the American Statistical
Association, 96(456), 1348–1360. https://doi.org/10.1198/016214501753382273
\item Hoerl, A. E., & Kennard, R. W. (1970). Ridge Regression: Biased Estimation
for Nonorthogonal Problems. Technometrics, 12(1), 55–67. https://doi.org/10.1080/00401706.1970.10488634
\item Tibshirani, R. (1996). Regression shrinkage and selection via the lasso.
Journal of the Royal Statistical Society. Series B (Methodological), 58(1), 267–288.
\item Zhang, C.-H. (2010). Nearly unbiased variable selection under minimax concave penalty.
The Annals of Statistics, 38(2), 894–942. https://doi.org/10.1214/09-AOS729
\item Zhang, T. (2010). Analysis of Multi-stage Convex Relaxation for Sparse Regularization.
Journal of Machine Learning Research, 11, 1081–1107.
\item Zou, H. (2006). The adaptive lasso and its oracle properties. Journal of the
American Statistical Association, 101(476), 1418–1429. https://doi.org/10.1198/016214506000000735
\item Zou, H., & Hastie, T. (2005). Regularization and variable selection via the
elastic net. Journal of the Royal Statistical Society: Series B, 67(2), 301–320.
https://doi.org/10.1111/j.1467-9868.2005.00503.x
}
}

\subsection{Optimizer}{
\subsection{GLMNET}{
\itemize{
\item Friedman, J., Hastie, T., & Tibshirani, R. (2010). Regularization paths for
generalized linear models via coordinate descent. Journal of Statistical
Software, 33(1), 1–20. https://doi.org/10.18637/jss.v033.i01
\item Yuan, G.-X., Ho, C.-H., & Lin, C.-J. (2012). An improved GLMNET for
l1-regularized logistic regression. The Journal of Machine Learning Research,
13, 1999–2030. https://doi.org/10.1145/2020408.2020421
}
}

\subsection{Variants of ISTA}{
\itemize{
\item Beck, A., & Teboulle, M. (2009). A Fast Iterative Shrinkage-Thresholding
Algorithm for Linear Inverse Problems. SIAM Journal on Imaging Sciences, 2(1),
183–202. https://doi.org/10.1137/080716542
\item Gong, P., Zhang, C., Lu, Z., Huang, J., & Ye, J. (2013). A general iterative
shrinkage and thresholding algorithm for non-convex regularized optimization problems.
Proceedings of the 30th International Conference on Machine Learning, 28(2)(2), 37–45.
\item Parikh, N., & Boyd, S. (2013). Proximal Algorithms. Foundations and
Trends in Optimization, 1(3), 123–231.
}
}

}

\subsection{Miscellaneous}{
\itemize{
\item Liang, X., Yang, Y., & Huang, J. (2018). Evaluation of structural relationships in autoregressive cross-lagged models under longitudinal approximate invariance: A Bayesian analysis. Structural Equation Modeling: A Multidisciplinary Journal, 25(4), 558–572. https://doi.org/10.1080/10705511.2017.1410706
\item Bauer, D. J., Belzak, W. C. M., & Cole, V. T. (2020). Simplifying the Assessment of Measurement Invariance over Multiple Background Variables: Using Regularized Moderated Nonlinear Factor Analysis to Detect Differential Item Functioning. Structural Equation Modeling: A Multidisciplinary Journal, 27(1), 43–55. https://doi.org/10.1080/10705511.2019.1642754
}
}
}

\section{Important Notes}{
THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
}

\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/jhorzek/lessSEM}
  \item \url{https://jhorzek.github.io/lessSEM/}
  \item Report bugs at \url{https://github.com/jhorzek/lessSEM/issues}
}

}
\author{
Jannik Orzek \href{mailto:orzek@mpib-berlin.mpg.de}{orzek@mpib-berlin.mpg.de}
}
\keyword{internal}
