% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_engine.R
\name{data_decode}
\alias{data_decode}
\alias{data_encode}
\title{Decode and encode text and binary data files}
\usage{
data_decode(data, encoding, as_text = FALSE, options = list())

data_encode(file, encoding, options = list(), output = NULL)
}
\arguments{
\item{data}{Encoded data as a character string}

\item{encoding}{Either \code{'base64'} or \code{'gpg'}}

\item{as_text}{A boolean indicating if decoded data should be treated as text
(\code{TRUE}) or binary (\code{FALSE}). Defaults to \code{FALSE}, meaning
binary.}

\item{options}{A list containing extra arguments for the encoding/decoding
functions. For \code{base64} encoding, \code{linewidth} (defaults to 64)
and and \code{newline} (defaults to \code{\link{platform.newline}()})
optional arguments are possible. For \code{gpg} encoding, see the
description below for details regarding the required \code{receiver} option
to define the key to use for encryption. For further details and
potentially other additional arguments, see the help of the corresponding
underlying encoding functions:
\code{\link[xfun:base64_encode]{base64_encode}} and
\code{\link[gpg]{gpg_encrypt}}.}

\item{file}{Path to file containing data to be encoded}

\item{output}{Path where encoded data is to be stored. Optional; if
\code{NULL} then encoded data will not be written to a file.}
}
\value{
Returns either the decoded data (\code{data_decode}) or the encoded
  data (\code{data_encode}).
}
\description{
These helper functions allow one to encode as text a binary or text data file
using either \code{base64} or \code{gpg} encoding (\code{data_encode}) and
decode text-encoded data back into its original binary or text format
(\code{data_decode}).
}
\details{
Encoding and decoding in \code{base64} format uses functionality from the
\code{\link[xfun:base64_encode]{xfun}} package, whereas encoding and decoding
using \code{gpg} uses functionality from the
\code{\link[gpg:gpg_encrypt]{gpg}} package. See those packages for more
details on the encoding and decoding process and setting up a \code{gpg}
keyring.

\code{data_encode} takes the name of a file containing the binary or text
data to be encoded and returns the encoded data as a character string. The
encoded data is returned silently to avoid outputing to the screen large
amounts of encoded data. If you want to visualize the encoded data, use the
\code{cat} function. For larger data files, set the \code{output} argument to
a path where the encoded data will be stored.

\code{data_encode} takes a character string of encoded data and returns
either a character string of decoded data (if \code{as_text=TRUE}) or a raw
vector of decoded binary data (if \code{as_text=FALSE}).

For both functions, the \code{options} input argument takes a list of
additional input arguments that are passed directly to the encoding or
decoding functions in the respective packages that handle the actual data
translation. See \code{\link[xfun:base64_encode]{base64_encode}} and
\code{\link[gpg]{gpg_encrypt}} for details.

For \code{gpg} encoding and decoding, in addition to installing the
\code{\link[gpg:gpg_encrypt]{gpg}} package, a \code{gpg} keyring must be
installed and properly configured. For encoding, the \code{receiver} and
potentially the \code{signer} arguments must be supplied as elements of the
\code{options} input argument.
}
\section{Functions}{
\itemize{
\item \code{data_decode()}: Returns decoded data as either a character string
(\code{as_text=TRUE}) or raw vector (\code{as_text=FALSE}).

\item \code{data_encode()}: Returns data encoded as a character string using
\code{base64} or \code{gpg} encoding.

}}
\examples{
# Use a temporary directory ----------------------------
owd = getwd()
td = tempdir()
setwd(td)

# Do some data encoding and decoding ------------------
library(knitrdata)

x = data.frame(a=1:5,b=letters[1:5])
write.csv(x,"test.csv")
saveRDS(x,"test.RDS")

enccsv = data_encode("test.csv","base64")
encrds = data_encode("test.RDS","base64")

csv = data_decode(enccsv,"base64",as_text=TRUE)
cat(csv)

rds = data_decode(encrds,"base64",as_text=FALSE)
writeBin(rds,"test_output.RDS")
y = readRDS("test_output.RDS")
y

params = list(run_gpg=FALSE)
if (requireNamespace("gpg") && params$run_gpg) {
  k = gpg::gpg_keygen("test","test@test.org")
  encgpg = data_encode("test.csv","gpg",options = list(receiver=k))

  cat(data_decode(encgpg,"gpg",as_text=TRUE))

  gpg::gpg_delete(k,secret=TRUE)
}

# Cleanup ------------------------------------
file.remove("test.csv","test.RDS","test_output.RDS")

setwd(owd)
}
\seealso{
See also \code{\link[xfun:base64_encode]{base64_encode}} and
  \code{\link[gpg]{gpg_encrypt}}, \code{\link{platform.newline}}.
}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\concept{decode encode}
