/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5module.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5MMprivate.h" 

static bool H5_ntzset = false;

#ifndef H5_HAVE_VASPRINTF

int
HDvasprintf(char **bufp, const char *fmt, va_list _ap)
{
    char  *buf;   
    size_t bufsz; 

    for (bufsz = 32; (buf = malloc(bufsz)) != NULL;) {
        int     ret;
        va_list ap;

        va_copy(ap, _ap);
        ret = vsnprintf(buf, bufsz, fmt, ap);
        va_end(ap);
        if (ret >= 0 && (size_t)ret < bufsz) {
            *bufp = buf;
            return ret;
        }
        free(buf);
        if (ret < 0)
            return ret;
        bufsz = (size_t)ret + 1;
    }
    return -1;
}
#endif 

#ifdef H5_HAVE_FCNTL
int
Pflock(int fd, int operation)
{

    struct flock flk;

    
    if (operation & LOCK_UN)
        flk.l_type = F_UNLCK;
    else if (operation & LOCK_SH)
        flk.l_type = F_RDLCK;
    else
        flk.l_type = F_WRLCK;

    
    flk.l_whence = SEEK_SET;
    flk.l_start  = 0;
    flk.l_len    = 0; 
    flk.l_pid    = 0; 

    
    if (fcntl(fd, F_SETLK, &flk) < 0)
        return -1;

    return 0;

} 
#endif 

int H5_ATTR_CONST
Nflock(int H5_ATTR_UNUSED fd, int H5_ATTR_UNUSED operation)
{
    
    return 0;
} 

time_t
H5_make_time(struct tm *tm)
{
    time_t the_time; 
#if defined(H5_HAVE_VISUAL_STUDIO)
    long timezone = 0;
#endif
    time_t ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(tm);

    
    if (!H5_ntzset) {
        tzset();
        H5_ntzset = true;
    }

    
    if ((time_t)-1 == (the_time = mktime(tm)))
        HGOTO_ERROR(H5E_INTERNAL, H5E_CANTCONVERT, FAIL, "badly formatted modification time message");

        
#if defined(H5_HAVE_TM_GMTOFF)
    
    the_time += tm->tm_gmtoff;
#elif defined(H5_HAVE_TIMEZONE)
#if defined(H5_HAVE_VISUAL_STUDIO)
    _get_timezone(&timezone);
#endif

    the_time -= timezone - (tm->tm_isdst ? 3600 : 0);
#else
    
    HGOTO_ERROR(H5E_INTERNAL, H5E_UNSUPPORTED, FAIL, "unable to obtain local timezone information");
#endif

    
    ret_value = the_time;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_WIN32_API

#define _W32_FT_OFFSET (116444736000000000ULL)

int
Wgettimeofday(struct timeval *tv, struct timezone *tz)
{
    union {
        unsigned long long ns100; 
        FILETIME           ft;
    } _now;

    static int tzsetflag;

    if (tv) {
        GetSystemTimeAsFileTime(&_now.ft);
        tv->tv_usec = (long)((_now.ns100 / 10ULL) % 1000000ULL);
        tv->tv_sec  = (long)((_now.ns100 - _W32_FT_OFFSET) / 10000000ULL);
    }

    if (tz) {
        if (!tzsetflag) {
            _tzset();
            tzsetflag = 1;
        }
        tz->tz_minuteswest = _timezone / 60;
        tz->tz_dsttime     = _daylight;
    }

    
    return 0;
} 

int
Wsetenv(const char *name, const char *value, int overwrite)
{
    
    if (!overwrite) {
#ifndef H5_HAVE_MINGW
        size_t  bufsize;
        errno_t err;

        err = getenv_s(&bufsize, NULL, 0, name);
        if (err || bufsize)
            return (int)err;
#else
        
        char *test = getenv(name);
        if (*test)
            return FAIL;
#endif
    }

    return (int)_putenv_s(name, value);
} 

#ifdef _MSC_VER
#pragma comment(lib, "advapi32.lib")
#endif

#ifdef H5_HAVE_WIN32_API
int
H5_get_win32_times(H5_timevals_t *tvs )
{
    static HANDLE        process_handle;
    ULARGE_INTEGER       kernel_start;
    ULARGE_INTEGER       user_start;
    FILETIME             KernelTime;
    FILETIME             UserTime;
    FILETIME             CreationTime;
    FILETIME             ExitTime;
    LARGE_INTEGER        counts_start;
    static LARGE_INTEGER counts_freq;
    static bool          is_initialized = false;
    BOOL                 err;

    assert(tvs);

    if (!is_initialized) {
        
        process_handle = GetCurrentProcess();
        err            = QueryPerformanceFrequency(&counts_freq);
        if (0 == err)
            return -1;
        is_initialized = true;
    } 

    

    err = GetProcessTimes(process_handle, &CreationTime, &ExitTime, &KernelTime, &UserTime);
    if (0 == err)
        return -1;

    
    kernel_start.HighPart = KernelTime.dwHighDateTime;
    kernel_start.LowPart  = KernelTime.dwLowDateTime;
    tvs->system           = (double)(kernel_start.QuadPart / 1.0E7F);

    user_start.HighPart = UserTime.dwHighDateTime;
    user_start.LowPart  = UserTime.dwLowDateTime;
    tvs->user           = (double)(user_start.QuadPart / 1.0E7F);

    

    err = QueryPerformanceCounter(&counts_start);
    if (0 == err)
        return -1;

    tvs->elapsed = (double)(counts_start.QuadPart) / (double)counts_freq.QuadPart;

    return 0;
} 
#endif

int
Wflock(int fd, int operation)
{

    HANDLE hFile;
    DWORD  dwFlags    = LOCKFILE_FAIL_IMMEDIATELY;
    DWORD  dwReserved = 0;
    
    DWORD nNumberOfBytesToLockLow  = MAXDWORD;
    DWORD nNumberOfBytesToLockHigh = MAXDWORD;
    
    OVERLAPPED overlapped = {0};

    
    if (INVALID_HANDLE_VALUE == (hFile = (HANDLE)_get_osfhandle(fd)))
        return -1;

    
    if (operation & LOCK_EX)
        dwFlags |= LOCKFILE_EXCLUSIVE_LOCK;

    
    if (operation & LOCK_UN) {
        if (0 ==
            UnlockFileEx(hFile, dwReserved, nNumberOfBytesToLockLow, nNumberOfBytesToLockHigh, &overlapped)) {
            
            if (GetLastError() != 158)
                return -1;
        }
    }
    else {
        if (0 == LockFileEx(hFile, dwFlags, dwReserved, nNumberOfBytesToLockLow, nNumberOfBytesToLockHigh,
                            &overlapped))
            return -1;
    }

    return 0;
} 

herr_t
H5_get_utf16_str(const char *s, wchar_t **wstring, uint32_t *win_error)
{
    int      nwchars = -1;   
    wchar_t *ret_s   = NULL; 

    
    if (0 == (nwchars = MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, s, -1, NULL, 0)))
        goto error;

    
    if (NULL == (ret_s = H5MM_calloc(sizeof(wchar_t) * (size_t)nwchars)))
        goto error;

    
    if (0 == MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, s, -1, ret_s, nwchars))
        goto error;

    *wstring = ret_s;

    return SUCCEED;

error:
    
    if (win_error)
        *win_error = (uint32_t)GetLastError();

    if (ret_s)
        H5MM_xfree((void *)ret_s);

    *wstring = NULL;

    return FAIL;
} 

int
Wopen(const char *path, int oflag, ...)
{
    uint32_t win_error        = 0;     
    wchar_t *wpath            = NULL;  
    herr_t   h5_ret           = FAIL;  
    char    *env              = NULL;  
    bool     prefer_code_page = false; 
    int      fd               = -1;    
    int      pmode            = 0;     

    
    oflag |= _O_BINARY;

    
    if (oflag & O_CREAT) {
        va_list vl;

        va_start(vl, oflag);
        pmode = va_arg(vl, int);
        va_end(vl);
    }

    
    env = getenv(HDF5_PREFER_WINDOWS_CODE_PAGE);
    if (env && (*env != '\0')) {
        if (0 == HDstrcasecmp(env, "true") || 0 == strcmp(env, "1"))
            prefer_code_page = true;
    }

    
    if (!prefer_code_page) {
        h5_ret = H5_get_utf16_str(path, &wpath, &win_error);
        if (h5_ret >= 0) {
            
            fd = _wopen(wpath, oflag, pmode);
        }
        else {
            if (ERROR_NO_UNICODE_TRANSLATION != win_error)
                goto done;

            fd = open(path, oflag, pmode);
        }
    }
    else
        fd = open(path, oflag, pmode);

done:
    H5MM_xfree(wpath);

    return fd;
} 

int
Wremove(const char *path)
{
    uint32_t win_error        = 0;     
    wchar_t *wpath            = NULL;  
    herr_t   h5_ret           = FAIL;  
    char    *env              = NULL;  
    bool     prefer_code_page = false; 
    int      ret              = -1;

    
    env = getenv(HDF5_PREFER_WINDOWS_CODE_PAGE);
    if (env && (*env != '\0')) {
        if (0 == HDstrcasecmp(env, "true") || 0 == strcmp(env, "1"))
            prefer_code_page = true;
    }

    
    if (!prefer_code_page) {
        h5_ret = H5_get_utf16_str(path, &wpath, &win_error);
        if (h5_ret >= 0) {
            
            ret = _wremove(wpath);
        }
        else {
            if (ERROR_NO_UNICODE_TRANSLATION != win_error)
                goto done;

            ret = remove(path);
        }
    }
    else
        ret = remove(path);

done:
    H5MM_xfree(wpath);

    return ret;
} 

#endif 

#define MAX_PATH_LEN 1024

herr_t
H5_build_extpath(const char *name, char **extpath )
{
    char  *full_path = NULL;    
    char  *cwdpath   = NULL;    
    char  *new_name  = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    assert(name);
    assert(extpath);

    
    *extpath = NULL;

    
    if (H5_CHECK_ABSOLUTE(name)) {
        if (NULL == (full_path = (char *)H5MM_strdup(name)))
            HGOTO_ERROR(H5E_INTERNAL, H5E_NOSPACE, FAIL, "memory allocation failed");
    }
    else {
        
        char  *retcwd;
        size_t name_len;
        int    drive;

        if (NULL == (cwdpath = (char *)H5MM_malloc(MAX_PATH_LEN)))
            HGOTO_ERROR(H5E_INTERNAL, H5E_NOSPACE, FAIL, "memory allocation failed");
        name_len = strlen(name) + 1;
        if (NULL == (new_name = (char *)H5MM_malloc(name_len)))
            HGOTO_ERROR(H5E_INTERNAL, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        if (H5_CHECK_ABS_DRIVE(name)) {
            drive  = toupper(name[0]) - 'A' + 1;
            retcwd = HDgetdcwd(drive, cwdpath, MAX_PATH_LEN);
            strncpy(new_name, &name[2], name_len);
        }
        
        else if (H5_CHECK_ABS_PATH(name) && (0 != (drive = HDgetdrive()))) {
            snprintf(cwdpath, MAX_PATH_LEN, "%c:%c", (drive + 'A' - 1), name[0]);
            retcwd = cwdpath;
            strncpy(new_name, &name[1], name_len);
        }
        
        else {
            retcwd = HDgetcwd(cwdpath, MAX_PATH_LEN);
            strncpy(new_name, name, name_len);
        }

        if (retcwd != NULL) {
            size_t cwdlen;
            size_t path_len;

            cwdlen = strlen(cwdpath);
            if (cwdlen == 0)
                HGOTO_ERROR(H5E_INTERNAL, H5E_BADVALUE, FAIL, "cwd length is zero");
            path_len = cwdlen + strlen(new_name) + 2;
            if (NULL == (full_path = (char *)H5MM_malloc(path_len)))
                HGOTO_ERROR(H5E_INTERNAL, H5E_NOSPACE, FAIL, "memory allocation failed");

            
            strncpy(full_path, cwdpath, path_len);
            full_path[path_len - 1] = '\0';

            if (!H5_CHECK_DELIMITER(cwdpath[cwdlen - 1]))
                strncat(full_path, H5_DIR_SEPS, path_len - (cwdlen + 1));
            strncat(full_path, new_name, path_len - (cwdlen + 1) - strlen(H5_DIR_SEPS));
        }
    }

    
    if (full_path) {
        char *ptr = NULL;

        H5_GET_LAST_DELIMITER(full_path, ptr)
        assert(ptr);
        *++ptr   = '\0';
        *extpath = full_path;
    }

done:
    if (cwdpath)
        H5MM_xfree(cwdpath);
    if (new_name)
        H5MM_xfree(new_name);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5_combine_path(const char *path1, const char *path2, char **full_name )
{
    size_t path1_len = 0;       
    size_t path2_len;           
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    assert(path2);

    if (path1)
        path1_len = strlen(path1);
    path2_len = strlen(path2);

    if (path1 == NULL || *path1 == '\0' || H5_CHECK_ABSOLUTE(path2)) {

        
        if (NULL == (*full_name = (char *)H5MM_strdup(path2)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    } 
    else if (H5_CHECK_ABS_PATH(path2)) {

        
        if (H5_CHECK_ABSOLUTE(path1) || H5_CHECK_ABS_DRIVE(path1)) {
            
            if (NULL == (*full_name = (char *)H5MM_malloc(path2_len + 3)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate path2 buffer");
            snprintf(*full_name, (path2_len + 3), "%c:%s", path1[0], path2);
        } 
        else {
            
            if (NULL == (*full_name = (char *)H5MM_strdup(path2)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
        } 

    } 
    else {

        
        if (NULL ==
            (*full_name = (char *)H5MM_malloc(path1_len + path2_len + 2 +
                                              2))) 
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate filename buffer");

        
        snprintf(*full_name, (path1_len + path2_len + 2 + 2), "%s%s%s",
                 path1, 
                 (H5_CHECK_DELIMITER(path1[path1_len - 1]) ? "" : H5_DIR_SEPS), path2);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5_nanosleep(uint64_t nanosec)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

#ifdef H5_HAVE_WIN32_API
    DWORD dwMilliseconds = (DWORD)ceil(nanosec / 1.0e6);

    
    SleepEx(dwMilliseconds, false);

#else

    const uint64_t  nanosec_per_sec = 1000 * 1000L * 1000;
    struct timespec sleeptime; 

    
    sleeptime.tv_sec  = (time_t)(nanosec / nanosec_per_sec);
    sleeptime.tv_nsec = (long)(nanosec % nanosec_per_sec);

    
    while (HDnanosleep(&sleeptime, &sleeptime) == -1) {
        
        if (errno != EINTR)
            break;
    }
#endif

    FUNC_LEAVE_NOAPI_VOID
} 

#ifdef H5_HAVE_WIN32_API

#define H5_WIN32_ENV_VAR_BUFFER_SIZE 32767

herr_t
H5_expand_windows_env_vars(char **env_var)
{
    long   n_chars   = 0;
    char  *temp_buf  = NULL;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    if (NULL == (temp_buf = (char *)H5MM_calloc((size_t)H5_WIN32_ENV_VAR_BUFFER_SIZE)))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for expanded path");

    
    if ((n_chars = ExpandEnvironmentStringsA(*env_var, temp_buf, H5_WIN32_ENV_VAR_BUFFER_SIZE)) >
        H5_WIN32_ENV_VAR_BUFFER_SIZE)
        HGOTO_ERROR(H5E_PLUGIN, H5E_NOSPACE, FAIL, "expanded path is too long");

    if (0 == n_chars)
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "failed to expand path");

    *env_var = (char *)H5MM_xfree(*env_var);
    *env_var = temp_buf;

done:
    if (FAIL == ret_value && temp_buf)
        temp_buf = (char *)H5MM_xfree(temp_buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

char *
H5_strndup(const char *s, size_t n)
{
    size_t len;
    char  *ret_value = NULL;

    FUNC_ENTER_NOAPI_NOINIT

    if (!s)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "string cannot be NULL");

    for (len = 0; len < n && s[len] != '\0'; len++)
        ;

    if (NULL == (ret_value = H5MM_malloc(len + 1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "can't allocate buffer for string");

    H5MM_memcpy(ret_value, s, len);
    ret_value[len] = '\0';

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

char *
Wstrcasestr_wrap(const char *haystack, const char *needle)
{
    if (needle && !*needle)
        return (char *)haystack;
    else
        return StrStrIA(haystack, needle);
}
#endif 

herr_t
H5_dirname(const char *path, char **dirname)
{
    char  *sep;
    char  *out       = NULL;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NOINIT

    if (!path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "path can't be NULL");
    if (!dirname)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dirname can't be NULL");

    if (NULL == (sep = strrchr(path, H5_DIR_SEPC))) {
        
        out = H5MM_strdup(".");
    }
    else if (sep == path) {
        
        out = H5MM_strdup(H5_DIR_SEPS);
    }
    else {
        if (sep[1] == '\0') {
            
            while (sep != path && sep[-1] == H5_DIR_SEPC)
                sep--;

            if (sep == path) {
                
                out = H5MM_strdup(H5_DIR_SEPS);
                sep = NULL;
            }
            else {
                
                while (sep != path && sep[-1] != H5_DIR_SEPC)
                    sep--;

                if (sep == path) {
                    
                    out = H5MM_strdup(".");
                    sep = NULL;
                }
            }
        }

        if (sep) {
            ptrdiff_t len;

            
            while (sep != path && sep[-1] == H5_DIR_SEPC)
                sep--;

            if (sep == path)
                
                out = H5MM_strdup(H5_DIR_SEPS);
            else {
                
                len = sep - path;
                assert(len >= 0);

                out = H5MM_strndup(path, (size_t)len);
            }
        }
    }

    if (NULL == out)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't allocate buffer for dirname");

    *dirname = out;

done:
    if (FAIL == ret_value) {
        H5MM_free(out);
        if (dirname)
            *dirname = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5_basename(const char *path, char **basename)
{
    const char *sep;
    char       *out       = NULL;
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NOINIT

    if (!path)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "path can't be NULL");
    if (!basename)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "basename can't be NULL");

    if (NULL == (sep = strrchr(path, H5_DIR_SEPC))) {
        if (*path == '\0')
            
            out = H5MM_strdup(".");
        else
            
            out = H5MM_strdup(path);
    }
    else if (sep == path) {
        if (sep[1] == '\0')
            
            out = H5MM_strdup(H5_DIR_SEPS);
        else
            
            out = H5MM_strdup(sep + 1);
    }
    else {
        if (sep[1] != '\0')
            
            out = H5MM_strdup(sep + 1);
        else {
            

            
            while (sep != path && sep[-1] == H5_DIR_SEPC)
                sep--;

            if (sep == path)
                
                out = H5MM_strdup(H5_DIR_SEPS);
            else {
                const char *c_ptr = sep;
                ptrdiff_t   len;

                
                while (c_ptr != path && c_ptr[-1] != H5_DIR_SEPC)
                    c_ptr--;

                len = sep - c_ptr;
                assert(len >= 0);

                out = H5MM_strndup(c_ptr, (size_t)len);
            }
        }
    }

    if (NULL == out)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't allocate buffer for basename");

    *basename = out;

done:
    if (FAIL == ret_value) {
        H5MM_free(out);
        if (basename)
            *basename = NULL;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

int         H5_opterr = 1; 
int         H5_optind = 1; 
const char *H5_optarg;     

int
H5_get_option(int argc, const char *const *argv, const char *opts, const struct h5_long_options *l_opts)
{
    static int sp      = 1;   
    int        optchar = '?'; 

    if (sp == 1) {
        
        if (H5_optind >= argc || argv[H5_optind][0] != '-' || argv[H5_optind][1] == '\0') {
            return EOF;
        }
        else if (strcmp(argv[H5_optind], "--") == 0) {
            H5_optind++;
            return EOF;
        }
    }

    if (sp == 1 && argv[H5_optind][0] == '-' && argv[H5_optind][1] == '-') {
        
        int        i;
        const char ch      = '=';
        char      *arg     = strdup(&argv[H5_optind][2]);
        size_t     arg_len = 0;

        H5_optarg = strchr(&argv[H5_optind][2], ch);
        arg_len   = strlen(&argv[H5_optind][2]);
        if (H5_optarg) {
            arg_len -= strlen(H5_optarg);
            H5_optarg++; 
        }
        arg[arg_len] = 0;

        for (i = 0; l_opts && l_opts[i].name; i++) {
            if (strcmp(arg, l_opts[i].name) == 0) {
                
                optchar = l_opts[i].shortval;

                if (l_opts[i].has_arg != no_arg) {
                    if (H5_optarg == NULL) {
                        if (l_opts[i].has_arg != optional_arg) {
                            if (H5_optind < (argc - 1))
                                if (argv[H5_optind + 1][0] != '-')
                                    H5_optarg = argv[++H5_optind];
                        }
                        else if (l_opts[i].has_arg == require_arg) {
                            if (H5_opterr)
                                Rfprintf(Rstderr, "%s: option required for \"--%s\" flag\n", argv[0], arg);

                            optchar = '?';
                        }
                    }
                }
                else {
                    if (H5_optarg) {
                        if (H5_opterr)
                            Rfprintf(Rstderr, "%s: no option required for \"%s\" flag\n", argv[0], arg);

                        optchar = '?';
                    }
                }
                break;
            }
        }

        if (l_opts && l_opts[i].name == NULL) {
            
            if (H5_opterr)
                Rfprintf(Rstderr, "%s: unknown option \"%s\"\n", argv[0], arg);

            optchar = '?';
        }

        H5_optind++;
        sp = 1;

        free(arg);
    }
    else {
        char *cp; 

        
        optchar = argv[H5_optind][sp];

        if (optchar == ':' || (cp = strchr(opts, optchar)) == 0) {
            if (H5_opterr)
                Rfprintf(Rstderr, "%s: unknown option \"%c\"\n", argv[0], optchar);

            
            if (argv[H5_optind][++sp] == '\0') {
                H5_optind++;
                sp = 1;
            }
            return '?';
        }

        if (*++cp == ':') {
            
            if (argv[H5_optind][sp + 1] != '\0') {
                
                H5_optarg = &argv[H5_optind++][sp + 1];
            }
            else if (++H5_optind >= argc) {
                if (H5_opterr)
                    Rfprintf(Rstderr, "%s: value expected for option \"%c\"\n", argv[0], optchar);

                optchar = '?';
            }
            else {
                
                H5_optarg = argv[H5_optind++];
            }

            sp = 1;
        }
        
        else if (*cp == '*') {
            
            H5_optind++;
            
            if ((H5_optind + 1) < argc) {
                if (argv[H5_optind][0] != '-') {
                    H5_optarg = argv[H5_optind++];
                }
                else {
                    H5_optarg = NULL;
                }
            }
            else {
                H5_optarg = NULL;
            }
        }
        else {
            
            if (argv[H5_optind][++sp] == '\0') {
                
                H5_optind++;
                sp = 1;
            }
            H5_optarg = NULL;
        }
    }

    
    return optchar;
}

char *
H5_strcasestr(const char *haystack, const char *needle)
{
    
    assert(haystack);
    assert(needle);

    
    do {
        const char *h = haystack;
        const char *n = needle;
        
        while (tolower(*h) == tolower(*n) && *n) {
            h++;
            n++;
        }
        
        if (*n == 0) {
            
            H5_WARN_CAST_AWAY_CONST_OFF
            return (char *)haystack;
            H5_WARN_CAST_AWAY_CONST_ON
        }
    } while (*haystack++);
    return 0;
} 

#if defined(H5_HAVE_WIN32_API) || defined(H5_HAVE_DARWIN) || (defined(__FreeBSD__) && __FreeBSD__ < 14)

typedef struct HDqsort_context_wrapper_t {
    int (*gnu_compar)(const void *, const void *, void *);
    void *gnu_arg;
} HDqsort_context_wrapper_t;

static int
HDqsort_context_wrapper_func(void *wrapper_arg, const void *a, const void *b)
{
    HDqsort_context_wrapper_t *w = (HDqsort_context_wrapper_t *)wrapper_arg;
    return w->gnu_compar(a, b, w->gnu_arg);
}

herr_t
HDqsort_context(void *base, size_t nel, size_t size, int (*compar)(const void *, const void *, void *),
                void *arg)
{
    HDqsort_context_wrapper_t wrapper;
    wrapper.gnu_compar = compar;
    wrapper.gnu_arg    = arg;
#if defined(H5_HAVE_WIN32_API)
    qsort_s(base, nel, size, HDqsort_context_wrapper_func, &wrapper);
#elif defined(H5_HAVE_DARWIN) || (defined(__FreeBSD__) && __FreeBSD__ < 14)
    
    qsort_r(base, nel, size, &wrapper, HDqsort_context_wrapper_func);
#endif
    return SUCCEED;
}
#endif

#ifndef H5_HAVE_QSORT_REENTRANT

typedef struct HDqsort_fallback_context_t {
    int (*gnu_compar)(const void *, const void *, void *);
    void *gnu_arg;
} HDqsort_fallback_context_t;

static HDqsort_fallback_context_t *HDqsort_fallback_global_ctx = NULL;

static int
HDqsort_fallback_wrapper(const void *a, const void *b)
{
    
    return HDqsort_fallback_global_ctx->gnu_compar(a, b, HDqsort_fallback_global_ctx->gnu_arg);
}

herr_t
HDqsort_fallback(void *base, size_t nel, size_t size, int (*compar)(const void *, const void *, void *),
                 void *arg)
{
    HDqsort_fallback_context_t ctx;

    ctx.gnu_compar = compar;
    ctx.gnu_arg    = arg;

    
    HDqsort_fallback_global_ctx = &ctx;

    qsort(base, nel, size, HDqsort_fallback_wrapper);

    
    HDqsort_fallback_global_ctx = NULL;

    return SUCCEED;
}

#endif 
