/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5TSpkg.h"     

#ifdef H5_HAVE_THREADS

static herr_t                  H5TS__pool_free(H5TS_pool_t *pool);
static H5TS_THREAD_RETURN_TYPE H5TS__pool_do(void *_pool);

H5FL_DEFINE_STATIC(H5TS_pool_t);

H5FL_SEQ_DEFINE_STATIC(H5TS_thread_t);

static herr_t
H5TS__pool_free(H5TS_pool_t *pool)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    assert(pool);

    
    for (unsigned u = 0; u < pool->num_threads; u++)
        if (H5_UNLIKELY(H5TS_thread_join(pool->threads[u], NULL) < 0))
            HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_semaphore_destroy(&pool->sem) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_destroy(&pool->queue_mutex) < 0))
        HGOTO_DONE(FAIL);

    
    if (pool->threads)
        H5FL_SEQ_FREE(H5TS_thread_t, pool->threads);
    H5FL_FREE(H5TS_pool_t, pool);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

static H5TS_THREAD_RETURN_TYPE
H5TS__pool_do(void *_pool)
{
    H5TS_pool_t      *pool      = (H5TS_pool_t *)_pool; 
    H5TS_thread_ret_t ret_value = (H5TS_thread_ret_t)0;

    
    while (1) {
        H5TS_pool_task_t *task; 

        
        if (H5_UNLIKELY(H5TS_semaphore_wait(&pool->sem) < 0))
            return ((H5TS_thread_ret_t)-1);

        
        if (H5_UNLIKELY(H5TS_mutex_lock(&pool->queue_mutex) < 0))
            return ((H5TS_thread_ret_t)-1);

        
        if (H5_LIKELY(pool->head)) {
            
            task = pool->head;
            if (task->next)
                pool->head = task->next;
            else
                pool->head = pool->tail = NULL;

            
            if (H5_UNLIKELY(H5TS_mutex_unlock(&pool->queue_mutex) < 0))
                return ((H5TS_thread_ret_t)-1);

            
            (*task->func)(task->ctx);

            
            free(task);
        }
        else {
            assert(pool->shutdown);

            
            if (H5_UNLIKELY(H5TS_mutex_unlock(&pool->queue_mutex) < 0))
                return ((H5TS_thread_ret_t)-1);

            break;
        }
    }

    return (ret_value);
} 

herr_t
H5TS_pool_create(H5TS_pool_t **pool, unsigned num_threads)
{
    H5TS_pool_t *new_pool = NULL; 
    unsigned     u;               
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == pool))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(0 == num_threads))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(NULL == (new_pool = H5FL_MALLOC(H5TS_pool_t))))
        HGOTO_DONE(FAIL);

    
    memset(new_pool, 0, sizeof(*new_pool));
    if (H5_UNLIKELY(H5TS_mutex_init(&new_pool->queue_mutex, H5TS_MUTEX_TYPE_PLAIN) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_semaphore_init(&new_pool->sem, 0) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(NULL == (new_pool->threads = H5FL_SEQ_MALLOC(H5TS_thread_t, num_threads))))
        HGOTO_DONE(FAIL);

    
    for (u = 0; u < num_threads; u++) {
        
        if (H5_UNLIKELY(H5TS_thread_create(&new_pool->threads[u], H5TS__pool_do, (void *)new_pool) < 0)) {
            
            new_pool->num_threads = u;
            HGOTO_DONE(FAIL);
        }
    }

    
    new_pool->num_threads = num_threads;

    
    *pool = new_pool;

done:
    if (H5_UNLIKELY(ret_value < 0))
        if (new_pool) {
            
            new_pool->shutdown = true;

            
            H5TS__pool_free(new_pool);
        }

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_pool_destroy(H5TS_pool_t *pool)
{
    bool   have_queue_mutex = false; 
    herr_t ret_value        = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == pool))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&pool->queue_mutex) < 0))
        HGOTO_DONE(FAIL);
    have_queue_mutex = true;

    
    pool->shutdown = true;

    
    if (H5_UNLIKELY(H5TS_mutex_unlock(&pool->queue_mutex) < 0))
        HGOTO_DONE(FAIL);
    have_queue_mutex = false;

    
    for (unsigned u = 0; u < pool->num_threads; u++)
        if (H5_UNLIKELY(H5TS_semaphore_signal(&pool->sem) < 0))
            HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS__pool_free(pool) < 0))
        HGOTO_DONE(FAIL);

done:
    
    if (H5_UNLIKELY(have_queue_mutex))
        if (H5_UNLIKELY(H5TS_mutex_unlock(&pool->queue_mutex) < 0))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif 
