/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5MMprivate.h" 

typedef struct {
    H5G_link_table_t *ltable;   
    size_t            curr_lnk; 
} H5G_iter_bt_t;

typedef struct {
    
    H5F_t      *file;            
    H5RS_str_t *grp_full_path_r; 
    const char *name;            
} H5G_iter_rm_t;

typedef struct {
    
    const char *name; 

    
    H5O_link_t *lnk;   
    bool       *found; 
} H5G_iter_lkp_t;

static herr_t H5G__compact_build_table_cb(const void *_mesg, unsigned idx, void *_udata);
static herr_t H5G__compact_build_table(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5_index_t idx_type,
                                       H5_iter_order_t order, H5G_link_table_t *ltable);
static herr_t H5G__compact_lookup_cb(const void *_mesg, unsigned H5_ATTR_UNUSED idx, void *_udata);

static herr_t
H5G__compact_build_table_cb(const void *_mesg, unsigned H5_ATTR_UNUSED idx, void *_udata)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_mesg; 
    H5G_iter_bt_t    *udata     = (H5G_iter_bt_t *)_udata;   
    herr_t            ret_value = H5_ITER_CONT;              

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);
    assert(udata->curr_lnk < udata->ltable->nlinks);

    
    if (NULL == H5O_msg_copy(H5O_LINK_ID, lnk, &(udata->ltable->lnks[udata->curr_lnk])))
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");

    
    udata->curr_lnk++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__compact_build_table(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5_index_t idx_type,
                         H5_iter_order_t order, H5G_link_table_t *ltable)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oloc);
    assert(linfo);
    assert(ltable);

    
    H5_CHECK_OVERFLOW(linfo->nlinks, hsize_t, size_t);
    ltable->nlinks = (size_t)linfo->nlinks;

    
    if (ltable->nlinks > 0) {
        H5G_iter_bt_t       udata; 
        H5O_mesg_operator_t op;    

        
        if ((ltable->lnks = (H5O_link_t *)H5MM_calloc(sizeof(H5O_link_t) * ltable->nlinks)) == NULL)
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

        
        udata.ltable   = ltable;
        udata.curr_lnk = 0;

        
        op.op_type  = H5O_MESG_OP_APP;
        op.u.app_op = H5G__compact_build_table_cb;
        if (H5O_msg_iterate(oloc, H5O_LINK_ID, &op, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "error iterating over link messages");

        
        if (H5G__link_sort_table(ltable, idx_type, order) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTSORT, FAIL, "error sorting link messages");
    } 
    else
        ltable->lnks = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_insert(const H5O_loc_t *grp_oloc, H5O_link_t *obj_lnk)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_oloc && grp_oloc->file);
    assert(obj_lnk);

    
    if (H5O_msg_create(grp_oloc, H5O_LINK_ID, 0, H5O_UPDATE_TIME, obj_lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_get_name_by_idx(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5_index_t idx_type,
                             H5_iter_order_t order, hsize_t idx, char *name, size_t name_size,
                             size_t *name_len)
{
    H5G_link_table_t ltable    = {0, NULL}; 
    herr_t           ret_value = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(oloc);

    
    if (H5G__compact_build_table(oloc, linfo, idx_type, order, &ltable) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create link message table");

    
    if (idx >= ltable.nlinks)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

    
    *name_len = strlen(ltable.lnks[idx].name);

    
    if (name) {
        strncpy(name, ltable.lnks[idx].name, MIN((*name_len + 1), name_size));
        if (*name_len >= name_size)
            name[name_size - 1] = '\0';
    } 

done:
    
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__compact_remove_common_cb(const void *_mesg, unsigned H5_ATTR_UNUSED idx, void *_udata)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_mesg; 
    H5G_iter_rm_t    *udata     = (H5G_iter_rm_t *)_udata;   
    herr_t            ret_value = H5_ITER_CONT;              

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    
    if (strcmp(lnk->name, udata->name) == 0) {
        
        if (H5G__link_name_replace(udata->file, udata->grp_full_path_r, lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get object type");

        
        HGOTO_DONE(H5_ITER_STOP);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_remove(const H5O_loc_t *oloc, H5RS_str_t *grp_full_path_r, const char *name)
{
    H5G_iter_rm_t udata;               
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(oloc && oloc->file);
    assert(name && *name);

    
    udata.file            = oloc->file;
    udata.grp_full_path_r = grp_full_path_r;
    udata.name            = name;

    
    if (H5O_msg_remove_op(oloc, H5O_LINK_ID, H5O_FIRST, H5G__compact_remove_common_cb, &udata, true) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete link message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_remove_by_idx(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5RS_str_t *grp_full_path_r,
                           H5_index_t idx_type, H5_iter_order_t order, hsize_t n)
{
    H5G_link_table_t ltable = {0, NULL};  
    H5G_iter_rm_t    udata;               
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(oloc && oloc->file);
    assert(linfo);

    
    if (H5G__compact_build_table(oloc, linfo, idx_type, order, &ltable) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create link message table");

    
    if (n >= ltable.nlinks)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "index out of bound");

    
    udata.file            = oloc->file;
    udata.grp_full_path_r = grp_full_path_r;
    udata.name            = ltable.lnks[n].name;

    
    if (H5O_msg_remove_op(oloc, H5O_LINK_ID, H5O_FIRST, H5G__compact_remove_common_cb, &udata, true) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete link message");

done:
    
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_iterate(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5_index_t idx_type,
                     H5_iter_order_t order, hsize_t skip, hsize_t *last_lnk, H5G_lib_iterate_t op,
                     void *op_data)
{
    H5G_link_table_t ltable    = {0, NULL}; 
    herr_t           ret_value = FAIL;      

    FUNC_ENTER_PACKAGE

    
    assert(oloc);
    assert(linfo);
    assert(op);

    
    if (H5G__compact_build_table(oloc, linfo, idx_type, order, &ltable) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create link message table");

    
    if ((ret_value = H5G__link_iterate_table(&ltable, skip, last_lnk, op, op_data)) < 0)
        HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");

done:
    
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__compact_lookup_cb(const void *_mesg, unsigned H5_ATTR_UNUSED idx, void *_udata)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_mesg; 
    H5G_iter_lkp_t   *udata     = (H5G_iter_lkp_t *)_udata;  
    herr_t            ret_value = H5_ITER_CONT;              

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    
    if (strcmp(lnk->name, udata->name) == 0) {
        if (udata->lnk) {
            
            if (NULL == H5O_msg_copy(H5O_LINK_ID, lnk, udata->lnk))
                HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");
        } 

        
        *udata->found = true;

        
        HGOTO_DONE(H5_ITER_STOP);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_lookup(const H5O_loc_t *oloc, const char *name, bool *found, H5O_link_t *lnk)
{
    H5G_iter_lkp_t      udata;               
    H5O_mesg_operator_t op;                  
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(name && *name);
    assert(found);
    assert(lnk && oloc->file);

    
    udata.name  = name;
    udata.lnk   = lnk;
    udata.found = found;

    
    op.op_type  = H5O_MESG_OP_APP;
    op.u.app_op = H5G__compact_lookup_cb;
    if (H5O_msg_iterate(oloc, H5O_LINK_ID, &op, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "error iterating over link messages");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__compact_lookup_by_idx(const H5O_loc_t *oloc, const H5O_linfo_t *linfo, H5_index_t idx_type,
                           H5_iter_order_t order, hsize_t n, H5O_link_t *lnk)
{
    H5G_link_table_t ltable    = {0, NULL}; 
    herr_t           ret_value = SUCCEED;   

    FUNC_ENTER_PACKAGE

    
    assert(oloc && oloc->file);
    assert(linfo);
    assert(lnk);

    
    if (H5G__compact_build_table(oloc, linfo, idx_type, order, &ltable) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create link message table");

    
    if (n >= ltable.nlinks)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "index out of bound");

    
    if (NULL == H5O_msg_copy(H5O_LINK_ID, &ltable.lnks[n], lnk))
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy link message");

done:
    
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 
