/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5MMprivate.h" 

#define H5G_NODE_VERS 1 

static herr_t H5G__cache_node_get_initial_load_size(void *udata, size_t *image_len);
static void  *H5G__cache_node_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5G__cache_node_image_len(const void *thing, size_t *image_len);
static herr_t H5G__cache_node_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5G__cache_node_free_icr(void *thing);

const H5AC_class_t H5AC_SNODE[1] = {{
    H5AC_SNODE_ID,                         
    "Symbol table node",                   
    H5FD_MEM_BTREE,                        
    H5AC__CLASS_NO_FLAGS_SET,              
    H5G__cache_node_get_initial_load_size, 
    NULL,                                  
    NULL,                                  
    H5G__cache_node_deserialize,           
    H5G__cache_node_image_len,             
    NULL,                                  
    H5G__cache_node_serialize,             
    NULL,                                  
    H5G__cache_node_free_icr,              
    NULL,                                  
}};

H5FL_EXTERN(H5G_node_t);

H5FL_SEQ_EXTERN(H5G_entry_t);

static herr_t
H5G__cache_node_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5F_t *f = (H5F_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(image_len);

    
    *image_len = (size_t)(H5G_NODE_SIZE(f));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5G__cache_node_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5F_t         *f         = (H5F_t *)_udata;         
    H5G_node_t    *sym       = NULL;                    
    const uint8_t *image     = (const uint8_t *)_image; 
    const uint8_t *image_end = image + len - 1;         
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(image);
    assert(len > 0);
    assert(f);
    assert(dirty);

    
    if (NULL == (sym = H5FL_CALLOC(H5G_node_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    sym->node_size = (size_t)(H5G_NODE_SIZE(f));
    if (NULL == (sym->entry = H5FL_SEQ_CALLOC(H5G_entry_t, (size_t)(2 * H5F_SYM_LEAF_K(f)))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5_SIZEOF_MAGIC, image_end))
        HGOTO_ERROR(H5E_SYM, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (memcmp(image, H5G_NODE_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, NULL, "bad symbol table node signature");
    image += H5_SIZEOF_MAGIC;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, image_end))
        HGOTO_ERROR(H5E_SYM, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (H5G_NODE_VERS != *image++)
        HGOTO_ERROR(H5E_SYM, H5E_VERSION, NULL, "bad symbol table node version");

    
    if (H5_IS_BUFFER_OVERFLOW(image, 1, image_end))
        HGOTO_ERROR(H5E_SYM, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    image++;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 2, image_end))
        HGOTO_ERROR(H5E_SYM, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(image, sym->nsyms);

    
    if (H5G__ent_decode_vec(f, &image, image_end, sym->entry, sym->nsyms) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTLOAD, NULL, "unable to decode symbol table entries");

    
    ret_value = sym;

done:
    if (!ret_value)
        if (sym && H5G__node_free(sym) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTFREE, NULL, "unable to destroy symbol table node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__cache_node_image_len(const void *_thing, size_t *image_len)
{
    const H5G_node_t *sym = (const H5G_node_t *)_thing; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(sym);
    assert(sym->cache_info.type == H5AC_SNODE);
    assert(image_len);

    *image_len = sym->node_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__cache_node_serialize(const H5F_t *f, void *_image, size_t len, void *_thing)
{
    H5G_node_t *sym       = (H5G_node_t *)_thing; 
    uint8_t    *image     = (uint8_t *)_image;    
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(image);
    assert(sym);
    assert(sym->cache_info.type == H5AC_SNODE);
    assert(len == sym->node_size);

    
    H5MM_memcpy(image, H5G_NODE_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += H5_SIZEOF_MAGIC;

    
    *image++ = H5G_NODE_VERS;

    
    *image++ = 0;

    
    UINT16ENCODE(image, sym->nsyms);

    
    if (H5G__ent_encode_vec(f, &image, sym->entry, sym->nsyms) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTENCODE, FAIL, "can't serialize");

    
    memset(image, 0, len - (size_t)(image - (uint8_t *)_image));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__cache_node_free_icr(void *_thing)
{
    H5G_node_t *sym       = (H5G_node_t *)_thing; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(sym);
    assert(sym->cache_info.type == H5AC_SNODE);

    
    if (H5G__node_free(sym) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to destroy symbol table node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
