/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 

H5FL_DEFINE_STATIC(H5FA_dblock_t);

H5FL_BLK_DEFINE(chunk_elmts);

H5FL_BLK_DEFINE(fa_page_init);

H5FA_dblock_t *
H5FA__dblock_alloc(H5FA_hdr_t *hdr)
{
    H5FA_dblock_t *dblock    = NULL; 
    H5FA_dblock_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->cparam.nelmts > 0);

    
    if (NULL == (dblock = H5FL_CALLOC(H5FA_dblock_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "memory allocation failed for fixed array data block");

    
    if (H5FA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    dblock->hdr = hdr;

    
    dblock->dblk_page_nelmts = (size_t)1 << hdr->cparam.max_dblk_page_nelmts_bits;

    
    if (hdr->cparam.nelmts > dblock->dblk_page_nelmts) {
        
        hsize_t npages = ((hdr->cparam.nelmts + dblock->dblk_page_nelmts) - 1) / dblock->dblk_page_nelmts;

        
        H5_CHECKED_ASSIGN(dblock->npages, size_t, npages, hsize_t);

        
        assert(dblock->npages > 0);

        
        dblock->dblk_page_init_size = (dblock->npages + 7) / 8;
        assert(dblock->dblk_page_init_size > 0);

        
        if (NULL == (dblock->dblk_page_init = H5FL_BLK_CALLOC(fa_page_init, dblock->dblk_page_init_size)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "memory allocation failed for page init bitmask");

        
        dblock->dblk_page_size = (dblock->dblk_page_nelmts * hdr->cparam.raw_elmt_size) + H5FA_SIZEOF_CHKSUM;

        
        if (0 == hdr->cparam.nelmts % dblock->dblk_page_nelmts)
            dblock->last_page_nelmts = dblock->dblk_page_nelmts;
        else
            dblock->last_page_nelmts = (size_t)(hdr->cparam.nelmts % dblock->dblk_page_nelmts);
    } 
    else {
        hsize_t dblk_size = hdr->cparam.nelmts * hdr->cparam.cls->nat_elmt_size;

        
        H5_CHECK_OVERFLOW(dblk_size,  hsize_t,  size_t);
        if (NULL == (dblock->elmts = H5FL_BLK_MALLOC(chunk_elmts, (size_t)dblk_size)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for data block element buffer");
    } 

    
    ret_value = dblock;

done:
    if (!ret_value)
        if (dblock && H5FA__dblock_dest(dblock) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FA__dblock_create(H5FA_hdr_t *hdr, bool *hdr_dirty)
{
    H5FA_dblock_t *dblock = NULL;     
    haddr_t        dblock_addr;       
    bool           inserted  = false; 
    haddr_t        ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr_dirty);

    
    if (NULL == (dblock = H5FA__dblock_alloc(hdr)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for fixed array data block");

    
    hdr->stats.dblk_size = dblock->size = H5FA_DBLOCK_SIZE(dblock);

    
    if (HADDR_UNDEF == (dblock_addr = H5MF_alloc(hdr->f, H5FD_MEM_FARRAY_DBLOCK, (hsize_t)dblock->size)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "file allocation failed for fixed array data block");
    dblock->addr = dblock_addr;

    
    if (!dblock->npages)
        
        if ((hdr->cparam.cls->fill)(dblock->elmts, (size_t)hdr->cparam.nelmts) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "can't set fixed array data block elements to class's fill value");

    
    if (H5AC_insert_entry(hdr->f, H5AC_FARRAY_DBLOCK, dblock_addr, dblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINSERT, HADDR_UNDEF, "can't add fixed array data block to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblock) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add fixed array entry as child of array proxy");
        dblock->top_proxy = hdr->top_proxy;
    } 

    
    *hdr_dirty = true;

    
    ret_value = dblock_addr;

done:

    if (!H5_addr_defined(ret_value))
        if (dblock) {
            
            if (inserted)
                if (H5AC_remove_entry(dblock) < 0)
                    HDONE_ERROR(H5E_FARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove fixed array data block from cache");

            
            if (H5_addr_defined(dblock->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_FARRAY_DBLOCK, dblock->addr, (hsize_t)dblock->size) < 0)
                HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to release fixed array data block");

            
            if (H5FA__dblock_dest(dblock) < 0)
                HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to destroy fixed array data block");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FA_dblock_t *
H5FA__dblock_protect(H5FA_hdr_t *hdr, haddr_t dblk_addr, unsigned flags)
{
    H5FA_dblock_t         *dblock = NULL; 
    H5FA_dblock_cache_ud_t udata;         
    H5FA_dblock_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(dblk_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.hdr       = hdr;
    udata.dblk_addr = dblk_addr;

    
    if (NULL ==
        (dblock = (H5FA_dblock_t *)H5AC_protect(hdr->f, H5AC_FARRAY_DBLOCK, dblk_addr, &udata, flags)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect fixed array data block, address = %llu",
                    (unsigned long long)dblk_addr);

    
    if (hdr->top_proxy && NULL == dblock->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblock) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, NULL,
                        "unable to add fixed array entry as child of array proxy");
        dblock->top_proxy = hdr->top_proxy;
    } 

    
    ret_value = dblock;

done:
    
    if (!ret_value)
        
        if (dblock &&
            H5AC_unprotect(hdr->f, H5AC_FARRAY_DBLOCK, dblock->addr, dblock, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect fixed array data block, address = %llu",
                        (unsigned long long)dblock->addr)

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblock_unprotect(H5FA_dblock_t *dblock, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    if (H5AC_unprotect(dblock->hdr->f, H5AC_FARRAY_DBLOCK, dblock->addr, dblock, cache_flags) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect fixed array data block, address = %llu",
                    (unsigned long long)dblock->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblock_delete(H5FA_hdr_t *hdr, haddr_t dblk_addr)
{
    H5FA_dblock_t *dblock    = NULL; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(dblk_addr));

    
    if (NULL == (dblock = H5FA__dblock_protect(hdr, dblk_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect fixed array data block, address = %llu",
                    (unsigned long long)dblk_addr);

    
    if (dblock->npages) {
        haddr_t dblk_page_addr; 
        size_t  u;              

        
        dblk_page_addr = dblk_addr + H5FA_DBLOCK_PREFIX_SIZE(dblock);

        
        for (u = 0; u < dblock->npages; u++) {
            
            
            if (H5AC_expunge_entry(hdr->f, H5AC_FARRAY_DBLK_PAGE, dblk_page_addr, H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTEXPUNGE, FAIL,
                            "unable to remove array data block page from metadata cache");

            
            dblk_page_addr += dblock->dblk_page_size;
        } 
    }     

done:
    
    if (dblock && H5FA__dblock_unprotect(dblock, H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG |
                                                     H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__dblock_dest(H5FA_dblock_t *dblock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    if (dblock->hdr) {
        
        if (dblock->elmts && !dblock->npages) {
            
            assert(dblock->hdr->cparam.nelmts > 0);
            dblock->elmts = H5FL_BLK_FREE(chunk_elmts, dblock->elmts);
        } 

        
        if (dblock->npages) {
            
            assert(dblock->dblk_page_init_size > 0);
            if (dblock->dblk_page_init)
                dblock->dblk_page_init = H5FL_BLK_FREE(fa_page_init, dblock->dblk_page_init);
        } 

        
        if (H5FA__hdr_decr(dblock->hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        dblock->hdr = NULL;
    } 

    
    assert(NULL == dblock->top_proxy);

    
    dblock = H5FL_FREE(H5FA_dblock_t, dblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
