## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = "centre"
)
options(rmarkdown.html_vignette.check_title = FALSE)

## ----setup, message=FALSE, warning=FALSE, echo=FALSE--------------------------
library(ggdibbler)
library(ggdist)
library(tidyverse)
library(distributional)
library(patchwork)
library(ggthemes)

## ----echo=FALSE, fig.width=8, fig.height=4------------------------------------
set.seed(1)
density_data <- data.frame(xmean = rnorm(15),
                           xse = rexp(15,3)) |>
  mutate(xdist = distributional::dist_normal(xmean, xse)) |>
  mutate(dist_group = factor(xdist)) |>
  group_by(dist_group) |>
  mutate(dist_group = cur_group_id()) |>
  ungroup()

p1 <- density_data |>
  ggplot(aes(xdist = xdist)) +
  stat_slab(alpha=0.5) +
  theme_tufte() +
  ggtitle("ggdist: uncertainty as signal")
  
p2 <- density_data |>
  ggplot(aes(x=xdist)) + 
  geom_density_sample(times=10) +
  theme_tufte() +
  ggtitle("ggdibbler: uncertainty as noise")

p1 | p2

## ----dodgebarchart, fig.width=12, fig.height=4--------------------------------

set.seed(10)
catdog <- tibble(
    DogOwner = sample(c(TRUE, FALSE), 25, replace=TRUE),
    CatOwner = sample(c(TRUE, FALSE), 25, replace=TRUE))

random_catdog <- catdog |>
  mutate(
    DogOwner = dist_bernoulli(0.1 + 0.8*DogOwner),
  )
    

p1 <- ggplot(catdog, aes(DogOwner)) + 
  geom_bar_sample(aes(fill = CatOwner), 
                  position = position_dodge(preserve="single"))+
  theme_few() +
  theme(legend.position="none", aspect.ratio = 1)+
  ggtitle("ggplot: dodge")


p2 <- ggplot(random_catdog, aes(DogOwner)) + 
  geom_bar_sample(aes(fill = CatOwner), times=30,
                  position = "dodge_dodge") +
  theme_few() +
  theme(legend.position="none", aspect.ratio = 1)+
  ggtitle("ggdibbler: dodge_dodge")

p3 <- ggplot(random_catdog, aes(DogOwner)) + 
  geom_bar_sample(aes(fill = CatOwner),  times=30,
                  position = position_dodge(preserve="single")) +
  theme_few() +
  theme(legend.position="none", aspect.ratio = 1)+
  ggtitle("ggdibbler: dodge")

p3 <- ggplot(random_catdog, aes(DogOwner)) + 
  geom_bar_sample(aes(fill = CatOwner),  times=30,
                  position = position_dodge(preserve="single")) +
  theme_few() +
  theme(legend.position="none", aspect.ratio = 1)+
  ggtitle("ggdibbler: dodge")

p1 | p3 | p2


## ----textplot, fig.width=16, fig.height=4-------------------------------------
set.seed(10)
textdata <- expand_grid(x = c(1,2,3,4,5), y= c(1,2,3,4,5)) |>
  mutate(
    z0 = sample(c(TRUE, FALSE), 25, replace=TRUE)
    ) |>
  mutate(
    z1 = dist_bernoulli(0.1 + 0.8*z0),
    z2 = dist_bernoulli(0.3 + 0.4*z0),
    z3 = dist_bernoulli(0.5),
  )

textplot <- function(var, alpha){
  ggplot(textdata, aes(x=x, y=y, lab=get(var))) +
  # ggplot(textdata, aes(x=x, y=y, lab=get(var))) +
           geom_text_sample(aes(label = after_stat(lab)), 
                            size=4, alpha=alpha, times=30) +
    scale_colour_viridis_d(option="rocket") + 
           #scale_colour_manual(values = c("steelblue", "firebrick")) +
           theme_few() +
           theme(aspect.ratio = 1, legend.position = "none") 
}
p0 <- textplot("z0", 1) +
  ggtitle("Point Estimate")
p1 <- textplot("z1", 0.05) +
  ggtitle("Binomial with p = 0.9 or 0.1")
p2 <- textplot("z2", 0.05) +
  ggtitle("Binomial with p = 0.3 or 0.7")
p3 <- textplot("z3", 0.05) +
  ggtitle("Binomial with p = 0.5")


(p3 | p2 | p1 | p0)

## ----tileplot, fig.width=16, fig.height=4-------------------------------------
tiledata <- expand_grid(x = c(1,2,3,4,5), y = c(1,2,3,4,5)) |>
  mutate(
    z0 = rnorm(25, 0, 10)
    ) |>
  mutate(
    z1 = dist_normal(z0, 2),
    z2 = dist_normal(z0, 8),
    z3 = dist_normal(z0, 30),
  )

p5 <- ggplot(tiledata, aes(x=x, y=y)) +
  geom_tile(aes(fill=z0), times=30) + 
  geom_tile(colour="white", fill=NA) + 
  theme_few() +
  scale_fill_viridis_c(option="rocket") + 
  theme(aspect.ratio = 1, legend.position = "none") +
  ggtitle("Point Estimate")

p6 <- ggplot(tiledata, aes(x=x, y=y)) +
  geom_tile_sample(aes(fill=z1), times=30) + 
  geom_tile(colour="white", fill=NA) + 
  theme_few() +
  scale_fill_viridis_c(option="rocket") + 
  theme(aspect.ratio = 1, legend.position = "none") +
  ggtitle("Normal with sigma = 2")

p7 <- ggplot(tiledata, aes(x=x, y=y)) +
  geom_tile_sample(aes(fill=z2), times=30) + 
  geom_tile(colour="white", fill=NA) + 
  theme_few() +
  scale_fill_viridis_c(option="rocket") + 
  theme(aspect.ratio = 1, legend.position = "none")  +
  ggtitle("Normal with sigma = 8")

p8 <- ggplot(tiledata, aes(x=x, y=y)) +
  geom_tile_sample(aes(fill=z3), times=30) + 
  geom_tile(colour="white", fill=NA) + 
  theme_few() +
  scale_fill_viridis_c(option="rocket") + 
  theme(aspect.ratio = 1, legend.position = "none")  +
  ggtitle("Normal with sigma = 20")

(p8 | p7 | p6 | p5)

## ----fuzzblur, fig.width=10, fig.height=5-------------------------------------
# examples from ggdibbler
p1 <-  ggplot(uncertain_mtcars, aes(x = mpg)) +
  geom_dotplot_sample(binwidth = 1.5, alpha=0.2) +
  theme(aspect.ratio = 1)+
  theme_few() +
  ggtitle("Blur in geom_dotplot")
p2 <- ggplot(uncertain_mpg, aes(cty, hwy)) +
  geom_count_sample(alpha=0.05, times=30) +
  theme(aspect.ratio = 1)+
  theme_few() +
  ggtitle("Fuzziness in geom_count")
p1 + p2

## ----include=FALSE, eval=FALSE------------------------------------------------
# library(spelling)
# qmd <- "B_ggdibbler-philosophy.Rmd"
# check_spelling <- spell_check_files(
#   qmd,
#   lang = "en_GB"
# )
# if (nrow(check_spelling) > 0) {
#   print(check_spelling)
#   stop("Check spelling in Qmd files!")
# }

