% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/taxonomic-searches.R
\name{taxonomic_searches}
\alias{taxonomic_searches}
\title{Look up taxon information}
\description{
\code{\link[=search_taxa]{search_taxa()}} allows users to look up taxonomic names, and ensure they are
being matched correctly, before downloading data from the specified
organisation.

By default, names are supplied as strings; but users can also specify
taxonomic levels in a search using a \code{data.frame} or \code{tibble}. This is useful
when the taxonomic \emph{level} of the name in question needs to be specified,
in addition to it's identity. For example, a common method is to use the
\code{scientificName} column to list a Latinized binomial, but it is also possible
to list these separately under \code{genus} and \code{specificEpithet} (respectively).
A more common use-case is to distinguish between homonyms by listing higher
taxonomic units, by supplying columns like \code{kingdom}, \code{phylum} or \code{class}.

\code{\link[=search_identifiers]{search_identifiers()}} allows users to look up matching taxonomic names using
their unique \code{taxonConceptID}. In the ALA, all records are associated with
an identifier that uniquely identifies the taxon to which that record belongs.
Once those identifiers are known, this function allows you to use them to
look up further information on the taxon in question. Effectively this is the
inverse function to \code{\link[=search_taxa]{search_taxa()}}, which takes names and provides
identifiers.

Note that when taxonomic look-up is required within a pipe, the equivalent
to \code{\link[=search_taxa]{search_taxa()}} is \code{\link[=identify.data_request]{identify()}} (or
\code{\link[=galah_identify]{galah_identify()}}). The equivalent to \code{\link[=search_identifiers]{search_identifiers()}} is to use
\code{\link[=filter.data_request]{filter()}} to filter by \code{taxonConceptId}.
}
\details{
\code{search_taxa()} returns the taxonomic match of a supplied text string, along
with the following information:
\itemize{
\item \code{search_term}: The search term used by the user. When multiple search
terms are provided in a tibble, these are displayed in this column,
concatenated using \verb{_}.
\item \code{scientific_name}: The taxonomic name matched to the provided search
term, to the lowest identified taxonomic rank.
\item \code{taxon_concept_id}: The unique taxonomic identifier.
\item \code{rank}: The taxonomic rank of the returned result.
\item \code{match_type}: (ALA only) The method of name matching used by the name
matching service. More information can be found on the
\href{https://github.com/AtlasOfLivingAustralia/ala-name-matching?tab=readme-ov-file#understanding-the-name-matching-algorithm}{name matching github repository}.
\item \code{issues}: Any errors returned by the name matching service
(e.g. homonym, indeterminate species match). More information can be found
on the \href{https://github.com/AtlasOfLivingAustralia/ala-name-matching?tab=readme-ov-file#error-types}{name matching github repository}.
\item \verb{taxonomic names} (e.g. \code{kingdom}, \code{phylum}, \code{class}, \code{order},
\code{family}, \code{genus})
}

When querying using \code{\link[=request_metadata]{request_metadata()}}, you have the option to pass
\code{\link[=select.metadata_request]{select()}} within the query. The
easiest way to do this is \code{select(everything())}, but for completeness, the
following additional fields are available:
\itemize{
\item \code{success}: Logical indicating success or failure of the search
\item \code{scientific_name_authorship}: Author and year for the name in question
\item \code{name_type}: Usually \code{"SCIENTIFIC"}
\item \code{lft} and \code{rgt}: Numeric indices for taxonomic lookups
\item \code{species_group} and \code{species_subgroup}: List-columns giving group names
\item \verb{_id} fields for \code{rank} and any taxonomic rank fields (\code{kingdom_id}, \code{phylum_id} etc.)
}
}
\examples{
\dontrun{
# Search using a single string. 
# Note that `search_taxa()` is not case sensitive
search_taxa("Reptilia")

# Search using multiple strings. 
# `search_taxa()` will return one row per taxon
search_taxa("reptilia", "mammalia")

# Search using more detailed strings with authorship information
search_taxa("Acanthocladium F.Muell")

# Specify taxonomic levels in a tibble using "specificEpithet"
search_taxa(tibble::tibble(
  class = "aves", 
  family = "pardalotidae", 
  genus = "pardalotus", 
  specificEpithet = "punctatus"))

# Specify taxonomic levels in a tibble using "scientificName"                    
search_taxa(tibble::tibble(
  family = c("pardalotidae", "maluridae"), 
  scientificName = c("Pardalotus striatus striatus", "malurus cyaneus")))

# Use OOP for the same effect
# `identify()` tells the code that we want to search for _taxonomic_ metadata.
request_metadata() |>
  identify("crinia") |>
  collect()

# This approach has the advantage that we can call `select()`
request_metadata() |>
  identify("crinia") |>
  select(everything()) |>
  collect()

# Look up a unique taxon identifier
search_identifiers(query = "https://id.biodiversity.org.au/node/apni/2914510")

# OOP process for identifiers uses `filter()`, not `identify()`
# In these cases the `field` argument is used to specify `type`
request_metadata() |>
  filter(identifier = "https://id.biodiversity.org.au/node/apni/2914510") |>
  select(everything()) |>
  collect()
}
}
\seealso{
\code{\link[=search_all]{search_all()}} for how to get names if taxonomic identifiers
are already known. \code{\link[=filter.data_request]{filter()}},
\code{\link[=select.data_request]{select()}},
\code{\link[=identify.data_request]{identify()}} and \code{\link[=geolocate]{geolocate()}} for ways
to restrict the information returned by \code{\link[=atlas_]{atlas_()}} functions.
}
