#' Generate a citation for occurrence data
#'
#' If a `tibble` containing occurrences was generated using galah (either via
#' \code{\link[=collect.data_request]{collect()}} or [atlas_occurrences()]), it 
#' will usually contain associated metadata stored in `attributes()` that can be 
#' used to build a citation for that dataset. This function simply extracts that 
#' information, formats it, then both invisibly returns the formatted citation
#' and prints it to the console.
#' @param data A `tibble` generated by [atlas_occurrences()] or similar
#' @return Invisibly returns a string containing the citation for that dataset.
#' Primarily called for the side-effect of printing this string to the console.
#' @examples \dontrun{
#' x <- galah_call() |>
#'     identify("Heleioporus") |>
#'     filter(year == 2022) |>
#'     collect()
#' atlas_citation(x)
#' }
#' @export
atlas_citation <- function(data) {
  # get basic information from file
  modified_date <- attributes(data)$modified_date
  if(is.null(modified_date)){
    modified_date <- Sys.Date() |>
      format("%e %B %Y") |>
      trimws()
  }
  doi <- attributes(data)$doi
  citation <- attributes(data)$citation
  # search_url <- attributes(data)$search_url
  
  # get existing citation info
  r_citation <- citation() |>
    print(style = 'text') |> 
    utils::capture.output() |>
    glue::glue_collapse(sep = " ")
  galah_citation <- c("Westgate M, Kellie D, Stevenson M & Newman P (2025):",
                      "_galah: Biodiversity Data from the GBIF Node Network_.",
                      "R package version 2.2.0.",
                      "doi: 10.32614/CRAN.package.galah") |>
    glue::glue_collapse(sep = " ")
  
  # ask users to cite galah and R
  suffix_text <-  glue::glue("
  
  Please consider citing R & galah, in addition to your dataset:
       
         {r_citation}
         
         {galah_citation}")
  
  
  # set case when DOI is missing
  if(!is.null(doi)) {
    # ALA
    if(stringr::str_detect(doi, "10.26197/ala.")){
      result <- glue::glue("
      The citation for this dataset is:
      
         Atlas of Living Australia ({modified_date}) Occurrence download {doi}
      
      {suffix_text}
         
       ")
      cli::cli_text(result)
      invisible(result)
    # GBIF
    }else if(stringr::str_detect(doi, "10.15468/dl.")){
      result <- glue::glue("
      The citation for this dataset is:
      
         GBIF.org ({modified_date}) GBIF Occurrence Download {doi}
      
      {suffix_text}
         
       ")
      cli::cli_text(result)
      invisible(result)
    # Unknown
    }else{
      bullets <- c(
        "The supplied DOI was not recognized.",
        i = "Please consider checking the atlas in question for their citation guidelines"
      )
      cli::cli_warn(bullets)
      invisible(bullets)
    }
  }else{
    if(!is.null(citation)){
      result <- glue::glue("
      The citation for this dataset is:
      
                        {citation}
           
       {suffix_text}
       ")
       cli::cli_text(result)
       invisible(result)
    }else{
      bullets <- c(
        "This dataset does not have any citation information attached.",
        i = "Please consider checking the atlas in question for their citation guidelines"
      )
      cli::cli_warn(bullets)
      invisible(bullets)
    }
  }
}
