% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corr_haltons.R
\name{corr_haltons}
\alias{corr_haltons}
\title{Generate Correlated Random Variables Using Halton or Scrambled Halton Draws}
\usage{
corr_haltons(
  means,
  cholesky = NULL,
  stdev = NULL,
  correlations = NULL,
  hdraws = NULL,
  ndraws = 500,
  scrambled = FALSE,
  dist = "normal",
  lower = -Inf,
  upper = Inf
)
}
\arguments{
\item{means}{A numeric vector of means for each variable.}

\item{cholesky}{A Cholesky decomposition matrix to introduce correlation.}

\item{stdev}{A numeric vector of standard deviations for each variable. If
provided, the function will use these values instead of the Cholesky
decomposition matrix (must also provide a correlation matrix if providing
standard deviations). Default is NULL.}

\item{correlations}{A correlation matrix to introduce correlation. If
provided, the function will use these values instead of the Cholesky
decomposition matrix (must also provide standard deviations). Default is
NULL.}

\item{hdraws}{A matrix of Halton or scrambled Halton draws. If provided, the
function will use these draws instead of generating new ones. Default is
NULL.}

\item{ndraws}{An integer specifying the number of values to simulate for each
variable. Default is 500.}

\item{scrambled}{A logical value indicating whether to use scrambled Halton
draws. Default is FALSE.}

\item{dist}{A character string specifying the distribution type. Options are
"normal" and "truncated_normal". Default is "normal".}

\item{lower}{A numeric value specifying the lower bound for truncated normal
distribution. Default is -Inf.}

\item{upper}{A numeric value specifying the upper bound for truncated normal
distribution. Default is Inf.}
}
\value{
A matrix with \code{N} columns and \code{ndraws} rows containing the
  simulated values for the correlated random variables.
}
\description{
This function generates \code{N} correlated random variables using Halton or
scrambled Halton draws. The function supports normal and truncated normal
distributions.
}
\examples{
# Define mean, correlation, and standard deviations
means <- c(3, 2, 0.9)
sdevs <- c(0.25,1.5,0.8)
CORR <- matrix(c(1, -0.3, 0.5, -0.3, 1, -0.2, 0.5, -0.2, 1), 3, 3)

# Create the Cholesky decomposition matrix and set values for ndraws, etc.
ndraws <- 5000
scrambled <- TRUE
dist <- "normal"

# simulated the data
simulated_data <- corr_haltons(means, stdev=sdevs, correlations=CORR,
                                ndraws=ndraws, scrambled=scrambled,
                                dist=dist)

# look at the mean, standard deviation, and correlation of the simulated data
apply(simulated_data, 2, mean)
apply(simulated_data, 2, sd)
cor(simulated_data)

# providing a cholesky decomposition matrix
dist <- "normal"
cholesky <- chol(cor2cov(CORR, sdevs))
simulated_data <- corr_haltons(means, cholesky=cholesky, ndraws=ndraws,
                                scrambled=scrambled, dist=dist)
apply(simulated_data, 2, mean)
apply(simulated_data, 2, sd)
cor(simulated_data)

# Truncated normal
dist <- "truncated_normal"
lower <- 0
upper <- 30
simulated_data <- corr_haltons(means, cholesky=cholesky, ndraws=ndraws,
                                scrambled=scrambled, dist=dist,
                                lower=lower, upper=upper)
apply(simulated_data, 2, mean)
apply(simulated_data, 2, sd)
cor(simulated_data)

}
