% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_chirps.R
\name{get_chirps}
\alias{get_chirps}
\alias{get_chirps.default}
\alias{get_chirps.SpatVector}
\alias{get_chirps.SpatRaster}
\alias{get_chirps.SpatExtent}
\alias{get_chirps.sf}
\alias{get_chirps.geojson}
\title{Get CHIRPS precipitation data}
\usage{
get_chirps(object, dates, server, ...)

\method{get_chirps}{default}(object, dates, server, as.matrix = FALSE, ...)

\method{get_chirps}{SpatVector}(object, dates, server = "CHC", as.raster = TRUE, ...)

\method{get_chirps}{SpatRaster}(
  object,
  dates,
  server = "CHC",
  as.matrix = TRUE,
  as.raster = FALSE,
  ...
)

\method{get_chirps}{SpatExtent}(object, dates, server = "CHC", as.raster = TRUE, ...)

\method{get_chirps}{sf}(object, dates, server, as.sf = FALSE, ...)

\method{get_chirps}{geojson}(object, dates, server, as.geojson = FALSE, ...)

\method{get_chirps}{SpatExtent}(object, dates, server = "CHC", as.raster = TRUE, ...)
}
\arguments{
\item{object}{input, an object of class \code{\link[base]{data.frame}} (or
any other object that can be coerced to \code{data.frame}),
\code{\link[terra]{SpatVector}}, \code{\link[terra]{SpatRaster}},
\code{\link[sf]{sf}} or \code{geojson}}

\item{dates}{a character of start and end dates in that order in the format
"YYYY-MM-DD"}

\item{server}{a character that represents the server source "CHC" or
"ClimateSERV"}

\item{...}{additional arguments passed to \code{\link[terra]{terra}}
or \code{\link[sf]{sf}} methods
See details}

\item{as.matrix}{logical, returns an object of class \code{matrix}}

\item{as.raster}{logical, returns an object of class
\code{\link[terra]{SpatRaster}}}

\item{as.sf}{logical, returns an object of class \code{\link[sf]{sf}}}

\item{as.geojson}{logical, returns an object of class \code{geojson}}
}
\value{
A matrix, raster or a data frame of \acronym{CHIRPS} data:
\describe{
  \item{id}{the index for the rows in \code{object}}
  \item{dates}{the dates from which \acronym{CHIRPS} was requested}
  \item{lon}{the longitude as provided in \code{object}}
  \item{lat}{the latitude as provided in \code{object}}
  \item{chirps}{the \acronym{CHIRPS} value in mm}
}
}
\description{
Get daily precipitation data from the "Climate Hazards Group". Two server
 sources are available. The first, "CHC" (default) is recommended for
 multiple data-points, while "ClimateSERV" is recommended when few
 data-points are required (~ 50).
}
\details{
Data description at
\url{https://data.chc.ucsb.edu/products/CHIRPS-2.0/README-CHIRPS.txt}

\strong{Additional arguments when using server = "CHC"}

\bold{resolution}: numeric, resolution of CHIRPS tiles either 0.05 (default)
 or 0.25 degrees

\strong{Additional arguments when using server = "ClimateSERV"}

\bold{dist}: numeric, buffer distance for each \code{object} coordinate

\bold{nQuadSegs}: integer, number of segments per buffer quadrant

\bold{operation}: supported operations for ClimateSERV are:
 \tabular{rll}{
 \bold{operation}      \tab    \tab \bold{value}\cr
 max                   \tab =  \tab 0\cr
 min                   \tab =  \tab 1\cr
 median                \tab =  \tab 2\cr
 sum                   \tab =  \tab 4\cr
 average               \tab =  \tab 5 (\emph{default value})\cr
 }
}
\note{
\code{get_chirps()} may return some warning messages given by
\code{\link[sf]{sf}}, please look \CRANpkg{sf} documentation for possible
issues.
}
\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library("chirps")
library("terra")

# Case 1: return as a data.frame
dates = c("2017-12-15","2017-12-31")
lonlat = data.frame(lon = c(-55.0281,-54.9857), lat = c(-2.8094, -2.8756))

r1 = get_chirps(lonlat, dates, server = "CHC")

# Case 2: return a matrix
r2 = get_chirps(lonlat, dates, server = "CHC", as.matrix = TRUE)

# Case 3: input SpatVector and return raster
f = system.file("ex/lux.shp", package = "terra")
v = vect(f)
r3 = get_chirps(v, dates, server = "CHC", as.raster = TRUE)

# Case 4: input SpatExtent and return a raster within the extent
area = ext(c(-66, -64, -6, -4))

dates = c("2017-12-15", "2017-12-31")

r4 = get_chirps(area, dates, server = "CHC")

# Case 5: using the server "ClimateSERV"
r5 = get_chirps(lonlat, dates, server = "ClimateSERV")

# Case 6: from "ClimateSERV" and return as a matrix
r6 = get_chirps(lonlat, dates, server = "ClimateSERV", as.matrix = TRUE)

\dontshow{\}) # examplesIf}
}
\references{
Funk C. et al. (2015). Scientific Data, 2, 150066.
 \cr\doi{10.1038/sdata.2015.66}
}
