% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/site_species_metrics.R
\name{site_species_metrics}
\alias{site_species_metrics}
\title{Calculate metrics for sites and species relative to bioregions and chorotypes}
\usage{
site_species_metrics(
  bioregionalization,
  bioregion_metrics = c("Specificity", "NSpecificity", "Fidelity", "IndVal", "NIndVal",
    "Rho"),
  bioregionalization_metrics = "P",
  data_type = "auto",
  cluster_on = "site",
  comat,
  similarity = NULL,
  include_cluster = FALSE,
  index = names(similarity)[3],
  verbose = TRUE
)
}
\arguments{
\item{bioregionalization}{A \code{bioregion.clusters} object.}

\item{bioregion_metrics}{A \code{character} vector or a single \code{character} string
specifying the metrics to compute for each cluster. Available metrics depend
on the type of clustering (see arg \code{cluster_on}):
\itemize{
\item{\strong{When sites are clustered into bioregions} (default case):
species-level metrics include \code{"Specificity"}, \code{"NSpecificity"},
\code{"Fidelity"}, \code{"IndVal"}, \code{"NIndVal"}, \code{"Rho"}, and \code{"CoreTerms"}.
Site-level metrics include \code{"Richness"}, \code{"Rich_Endemics"},
\code{"Prop_Endemics"}, \code{"MeanSim"}, and \code{"SdSim"}.}
\item{\strong{When species are clustered into chorotypes} (e.g., bipartite
network clustering): site-level metrics include \code{"Specificity"},
\code{"NSpecificity"}, \code{"Fidelity"}, \code{"IndVal"}, \code{"NIndVal"}, \code{"Rho"},
and \code{"CoreTerms"}.}
}
Use \code{"all"} to compute all available metrics. See Details for metric
descriptions.}

\item{bioregionalization_metrics}{A \code{character} vector or a single \code{character}
string specifying summary metrics computed across all clusters. These
metrics assess how an entity (species or site) is distributed across the
entire bioregionalization, rather than relative to each individual cluster:
\itemize{
\item{\code{"P"}: Participation coefficient measuring how evenly a species or
site is distributed across clusters (0 = restricted to one cluster,
1 = evenly spread).}
\item{\code{"Silhouette"}: How well a site fits its assigned bioregion compared
to the nearest alternative bioregion (requires similarity data).}
}
Use \code{"all"} to compute all available metrics.}

\item{data_type}{A \code{character} string specifying whether metrics should be
computed based on presence/absence (\code{"occurrence"}) or abundance values
(\code{"abundance"}). This affects how Specificity, Fidelity, IndVal, Rho and
CoreTerms are calculated:
\itemize{
\item{\code{"auto"} (default): Automatically detected from input data (\code{bioregionalization} and/or \code{comat}).}
\item{\code{"occurrence"}: Metrics based on presence/absence only.}
\item{\code{"abundance"}: Metrics weighted by abundance values.}
\item{\code{"both"}: Compute both versions of the metrics.}
}}

\item{cluster_on}{A \code{character} string specifying what was clustered in the
bioregionalization, which determines what types of metrics can be computed:
\itemize{
\item{\code{"site"} (default): Sites were clustered into bioregions. Metrics
describe how each \strong{species} is distributed across bioregions.}
\item{\code{"species"}: Species were clustered into chorotypes. Metrics describe
how each \strong{site} relates to chorotypes. Only available when species have
been assigned to clusters (e.g., bipartite network clustering).}
\item{\code{"both"}: Compute metrics for both perspectives. Only available
when both sites and species have cluster assignments.}
}}

\item{comat}{A site-species \code{matrix} with sites as rows and species as
columns. Values can be occurrence (1/0) or abundance. Required for most
metrics.}

\item{similarity}{A site-by-site similarity object from \code{\link[=similarity]{similarity()}} or
\code{\link[=dissimilarity_to_similarity]{dissimilarity_to_similarity()}}. Required only for similarity-based metrics
(\code{"MeanSim"}, \code{"SdSim"}, \code{"Silhouette"}).}

\item{include_cluster}{A \code{boolean} indicating whether to add an \code{Assigned}
column in the output, marking \code{TRUE} for rows where the site belongs to the
bioregion being evaluated. Useful for quickly identifying a site's own
bioregion. Default is \code{FALSE}.}

\item{index}{The name or number of the column to use as similarity.
By default, the third column name of \code{similarity} is used.}

\item{verbose}{A \code{boolean} indicating whether to
display progress messages. Set to \code{FALSE} to suppress these messages.}
}
\value{
A \code{list} containing one or more \code{data.frame} elements, depending on the
selected metrics and clustering type:

\strong{When sites are clustered (\code{cluster_on = "site"}):}
\itemize{
\item{\strong{species_bioregions}: Metrics for each species x bioregion
combination (e.g., Specificity, IndVal). One row per species x bioregion
pair.}
\item{\strong{species_bioregionalization}: Summary metrics for each species
across all bioregions (e.g., Participation coefficient). One row per
species.}
\item{\strong{site_bioregions}: Metrics for each site x bioregion combination
(e.g., MeanSim, Richness). One row per site x bioregion pair.}
\item{\strong{site_bioregionalization}: Summary metrics for each site
(e.g., Silhouette). One row per site.}
}

\strong{When species are clustered (\code{cluster_on = "species"}):}
\itemize{
\item{\strong{site_chorotypes}: Metrics for each site x chorotype combination
(e.g., Specificity, IndVal). One row per site x chorotype pair.}
\item{\strong{site_chorological}: Summary metrics for each site across all
chorotypes (e.g., Participation coefficient). One row per site.}
}

Note that if \code{bioregionalization} contains multiple partitions
(i.e., if \code{dim(bioregionalization$clusters) > 2}), a nested list will be
returned, with one sublist per partition.
}
\description{
This function computes metrics that quantify how species and sites relate
to clusters (bioregions or chorotypes). Depending on the type of clustering,
metrics can measure how species are distributed across bioregions (site
clusters), how sites relate to chorotypes (species clusters), or both.
}
\details{
This function computes metrics that characterize the relationship between
species, sites, and clusters. The available metrics depend on whether you
clustered sites (into bioregions) or species (into chorotypes).
\subsection{--- 1. Understanding the two perspectives ---}{
\itemize{
\item \strong{Bioregions} are clusters of sites with similar species composition.
\item \strong{Chorotypes} are clusters of species with similar distributions.
}

In general, the package is designed to cluster sites into bioregions.
However, it is possible to group species into clusters.
We call these species clusters 'chorotypes',
following conceptual definitions in the biogeographical literature, to
avoid any confusion in the calculation of metrics.

In some cases, such as bipartite network clustering, both species and sites
receive the same clusters. We maintain the name distinction in the
calculation of metrics - but remember that in this case
BIOREGION IDs = CHOROTYPE IDs.
The \code{cluster_on} argument determines
which perspective to use.
}

\subsection{--- 2. Metrics when sites are clustered (\code{cluster_on = "site"} or \code{cluster_on = "both"}) ---}{

\strong{Species-per-bioregion metrics} quantify how each species is distributed
across bioregions.

These metrics are derived from three core terms (\href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#metric-components}{see the online vignette for a visual diagram}):
\itemize{
\item \strong{n_sb}: Number of sites in bioregion \strong{b} where species
\strong{s} is present
\item \strong{n_s}: Total number of sites in which species \strong{s} is present.
\item \strong{n_b}: Total number of sites in bioregion \strong{b}.
}

Abundance version of these core terms can also be calculated when
\code{data_type = "abundance"} (or \code{data_type = "auto"} and
\verb{bioregionalization was based on abundance}):
\itemize{
\item \strong{w_sb}: Sum of abundances of species \strong{s} in sites of bioregion \strong{b}.
\item \strong{w_s}: Total abundance of species \strong{s}.
\item \strong{w_b}: Total abundance of all species present in sites of bioregion \strong{b}.
}

The species-per-bioregion metrics are (click on metric names to access formulas):
\itemize{
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#specificity-occurrence}{\strong{Specificity}}:
Fraction of a species' occurrences found in a given bioregion
(De Cáceres & Legendre 2009). A value of
1 means the species occurs only in that bioregion.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#nspecificity-occurrence}{\strong{NSpecificity}}:
Normalized specificity that accounts for differences in bioregion size
(De Cáceres & Legendre 2009).
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#fidelity-occurrence}{\strong{Fidelity}}:
Fraction of sites in a bioregion where the species occurs
(De Cáceres & Legendre 2009). A value of 1
means the species is present in all sites of that bioregion.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#indval-occurrence}{\strong{IndVal}}:
Indicator Value = Specificity × Fidelity
(De Cáceres & Legendre 2009). High values identify species
that are both restricted to and frequent within a bioregion.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#nindval-occurrence}{\strong{NIndVal}}:
Normalized IndVal accounting for bioregion size
(De Cáceres & Legendre 2009).
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#rho-occurrence}{\strong{Rho}}:
Standardized contribution index comparing observed vs. expected
co-occurrence under random association (Lenormand 2019).
\item \strong{CoreTerms}: Raw counts (n, n_b, n_s, n_sb) for custom calculations.
}

These metrics can be found in the output slot \code{species_bioregions}.

\strong{Site-per-bioregion metrics} characterize sites relative to bioregions:
\itemize{
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#diversity-endemicity-site-metrics}{\strong{Richness}}:
Number of species in the site.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#diversity-endemicity-site-metrics}{\strong{Rich_Endemics}}:
Number of species in the site that are endemic to one bioregion.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#diversity-endemicity-site-metrics}{\strong{Prop_Endemics}}:
Proportion of endemic species in the site.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#meansim}{\strong{MeanSim}}:
Mean similarity of a site to all sites in each bioregion.
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#sdsim}{\strong{SdSim}}:
Standard deviation of similarity values.
}

These metrics can be found in the output slot \code{site_bioregions}.

\strong{Summary metrics across the whole bioregionalization:}

These metrics summarize how an entity (species or site) is distributed
across all clusters, rather than in relation to each individual cluster.

\emph{Species-level summary metric:}
\itemize{
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#p-occurrence-1}{\strong{P}}
(Participation): Evenness of species distribution across bioregions
(Denelle et al. 2020). Found in output slot \code{species_bioregionalization}.
}

\emph{Site-level summary metric:}
\itemize{
\item \href{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html#silhouette}{\strong{Silhouette}}:
How well a site fits its assigned bioregion vs. the nearest alternative
(Rousseeuw 1987). Found in output slot \code{site_bioregionalization}.
}
}

\subsection{--- 3. Metrics when species are clustered (\code{cluster_on = "species"} or \code{cluster_on = "both"}) ---}{

\strong{Site-per-chorotype metrics} quantify how each site relates to species
clusters (chorotypes).

The same metrics as above (Specificity, Fidelity,
IndVal, etc.) can be computed, but their interpretation is inverted. These
metrics are based on the following core terms:
\itemize{
\item \strong{n_gc}: Number of species belonging to chorotype \strong{c} that are present
in site \strong{g}.
\item \strong{n_g}: Total number of species present in site \strong{g}.
\item \strong{n_c}: Total number of species belonging to chorotype \strong{c}.
}

Abundance version of these core terms can also be calculated when
\code{data_type = "abundance"} (or \code{data_type = "auto"} and
\verb{bioregionalization was based on abundance}).

Their interpretation changes, for example:
\itemize{
\item \strong{Specificity}: Fraction of a site's species belonging to a chorotype.
\item \strong{Fidelity}: Fraction of a chorotype's species present in the site.
\item \strong{IndVal}: Indicator value for site-chorotype associations.
\item \strong{P}: Evenness of sites across chorotypes
}
}
}
\note{
If \code{data_type = "auto"}, the choice between occurrence- or abundance-
based metrics will be determined automatically from the input data, and a
message will explain the choice made.

Strict matching between entity IDs (site and species IDs) in
\code{bioregionalization} and in \code{comat} / \code{similarity} is required.
}
\examples{
data(fishmat)

fishsim <- similarity(fishmat, metric = "Jaccard")

bioregionalization <- hclu_hierarclust(similarity_to_dissimilarity(fishsim),
                                       index = "Jaccard",
                                       method = "average",
                                       randomize = TRUE,
                                       optimal_tree_method = "best",
                                       n_clust = c(1,2,3),
                                       verbose = FALSE)
                                     
ind <- site_species_metrics(bioregionalization = bioregionalization,
                             bioregion_metrics = "all",
                             bioregionalization_metrics = "all",
                             data_type = "auto",
                             cluster_on = "site",
                             comat = fishmat,
                             similarity = fishsim,
                             include_cluster = TRUE,
                             index = 3,
                             verbose = TRUE)
                                     
}
\references{
De Cáceres M & Legendre P (2009) Associations between species and groups of
sites: indices and statistical inference. \emph{Ecology} 90, 3566--3574.

Denelle P, Violle C & Munoz F (2020) Generalist plants are more competitive
and more functionally similar to each other than specialist plants: insights
from network analyses. \emph{Journal of Biogeography} 47, 1922–-1933.

Lenormand M, Papuga G, Argagnon O, Soubeyrand M, Alleaume S & Luque S (2019)
Biogeographical network analysis of plant species distribution in the
Mediterranean region. \emph{Ecology and Evolution} 9, 237--250.

Rousseeuw PJ (1987) Silhouettes: A graphical aid to the interpretation and
validation of cluster analysis. \emph{Journal of Computational and Applied
Mathematics} 20, 53--65.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a5_2_summary_metrics.html}.

Associated functions:
\link{bioregion_metrics} \link{bioregionalization_metrics}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com})
}
