% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curve_projection.R
\name{curve_projection}
\alias{curve_projection}
\alias{summary.curve_projection}
\alias{coef.curve_projection}
\alias{vcov.curve_projection}
\alias{anova.curve_projection}
\title{Project an effect curve onto a simpler model}
\usage{
curve_projection(x, model, transform = TRUE)

\method{summary}{curve_projection}(
  object,
  conf_level = 0.95,
  null = 0,
  df = NULL,
  ci.type = "perc",
  subset = NULL,
  ...
)

\method{coef}{curve_projection}(object, ...)

\method{vcov}{curve_projection}(object, ...)

\method{anova}{curve_projection}(object, object2, df = NULL, ...)
}
\arguments{
\item{x}{an \code{\link{effect_curve}} object; the output of a call to \code{\link[=adrf]{adrf()}}.}

\item{model}{the projection model to be fit. Can be a one-sided formula corresponding to the projection model or one of the following strings: \code{"flat"}, \code{"linear"}, \code{"quadratic"}, \code{"cubic"}.}

\item{transform}{whether to compute the projection using a transformation of the linear predictor. Allowable options include \code{TRUE}, \code{FALSE}, or a function specifying a transformation (of which the inverse is used as the inverse link of the projection model). Ignored unless \code{object} is an ADRF. See Details.}

\item{object, object2}{a \code{curve_projection} object; the output of a call to \code{curve_projection()}.}

\item{conf_level}{the desired confidence level. Set to 0 to omit confidence intervals. Default is .95.}

\item{null}{the null value for hypothesis test. Default is 0. Set to \code{NA} to omit tests.}

\item{df}{the "denominator" degrees of freedom to use for the test. Default is to use the residual degrees of freedom from the original model if it is a linear model (in which case an F-test is used) and \code{Inf} otherwise (in which case a \eqn{\chi^2} test is used).}

\item{ci.type}{string; when bootstrapping or Bayesian inference is used in the original effect curve, which type of confidence interval is to be computed. For bootstrapping, allowable options include \code{"perc"} for percentile intervals, \code{"wald"} for Wald intervals, and other options allowed by \pkgfun{fwb}{summary.fwb}. When \code{simultaneous = TRUE}, only \code{"perc"} and \code{"wald"} are allowed. For Bayesian models, allowable options include \code{"perc"} for equi-tailed intervals and \code{"wald"} for Wald intervals. Default is \code{"perc"}. Ignored when bootstrapping is not used and the model is not Bayesian.}

\item{subset}{an optional logical expression indicating the subset of subgroups for which to compute the projection. Can only be used when \code{by} was supplied to the original call to \code{\link[=adrf]{adrf()}}, and only to refer to variables defining subgroups.}

\item{\dots}{ignored.}
}
\value{
\code{curve_projection()} returns an \code{curve_projection} object, which inherits from \code{\link{effect_curve}}. This object is a function that produces estimates of the effect curve projection when called with values of the treatment as inputs. See \code{\link{effect_curve}} for details on calling this function.

The coefficients and covariance matrix of the fitted projection should be extracted with \code{coef()} and \code{vcov()}, respectively. \code{summary()} produces the coefficients and quantities of interest for inference (test statistics, p-values, and confidence intervals). Using \code{plot()} on a \code{curve_projection} object plots the projection curve as if it was an effect curve; see \code{\link[=plot.effect_curve]{plot.effect_curve()}} for adding the projection curve to the plot of the original effect curve.
}
\description{
\code{curve_projection()} produces a projection of an estimated effect curve onto a specified linear model that is a function only of the treatment to act as a more interpretable summary of the original effect curve.
}
\details{
The projection model can be thought of as a linear regression of the effect curve estimates on the treatment. Whereas the original effect curve may be complicated and nonlinear, the projection model can be simple and easily interpretable, though it must be understood as a summary of the original effect curve. For example, the original ADRF might have been computed from an outcome model that involves treatment splines, covariates, and treatment-covariate interactions. Though the ADRF is a univariable function (i.e., of only the treatment), it isn't described by a single set of parameters. The linear projection of the ADRF, though, could be a simple linear model, described by an intercept and the slope on treatment. Though only a rough approximation to the ADRF, the linear projection may be more easily interpreted. This concept is described in Neugebauer and van der Laan (2007).

\code{curve_projection()} fits this projection model and accounts for the uncertainty in the estimates of the effect curve in computing uncertainty estimates for the projection model parameters. Because the true effect curve is continuous, the model is fit minimizing
\deqn{\int_{a_\text{lower}}^{a_\text{upper}}{\left(\hat{\theta}(a)- B(a)\hat{\mathbf{\beta}} \right)^2 da}}
where \eqn{\hat{\theta}(a)} is the effect curve estimate at treatment value \eqn{a}, \eqn{B(a)} is the basis function representation of \eqn{a} (i.e., as specified in \code{model}), and \eqn{\hat{\mathbf{\beta}}} is the vector of projection parameters to be estimated. This integral is approximated using a trapezoidal Riemann sum over the effect curve grid points.

The covariance of the projection parameters can be computed using the delta method applied to the estimated covariance of the original effect curve estimates. When bootstrapping or posterior inference are used, the projection is applied to each bootstrap or posterior draw, respectively.
\subsection{Transform}{

When \code{transform} is specified, the projection minimizes the distance between the original effect curve and the transformed linear predictor; that is, it minimizes

\deqn{\int_{a_\text{lower}}^{a_\text{upper}}{\left(\hat{\theta}(a)-f^{-1}\left( B(a) \hat{\mathbf{\beta}} \right) \right)^2 da}}

where \eqn{f^{-1}(y)} is the inverse of the transformation supplied to \code{transform} (i.e., corresponding to the inverse link function of a generalized linear model), essentially using nonlinear least squares (NLS) to estimate the effect curve projection. This make the coefficients in the projection model correspond to the coefficients on the linear predictor \eqn{B(a)\hat{\mathbf{\beta}}}. In this case, the projection is not simply a linear projection, but it may still be more interpretable than the original ADRF. For example, if the outcome model was originally fit using logistic regression and \code{transform = TRUE} in the call to \code{curve_projection()} with \code{model = "linear"}, the resulting projection would be a logistic curve governed by the intercept and slope of the linear predictor. See Examples for an example of this.

By default, \code{transform} is \code{TRUE}, which means that when the original outcome model had a \code{family} component (e.g., a generalized linear model) and an ADRF is supplied to \code{curve_projection()}, the link is automatically supplied to \code{transform} and the projection model will be a nonlinear function of the linear predictor. Set \code{transform} to \code{FALSE} to require that the projection curve be simply the linear predictor with no transformation. Note this can lead to invalid estimates when the outcome is bounded.
}

\subsection{Comparing projection models}{

\code{anova()} performs a Wald test comparing two nested projection models. The null hypothesis is that the simpler model is sufficient, i.e., that the coefficients on the terms in the larger model (supplied to \code{object}) that are not in the smaller model (supplied to \code{object2}) are all zero. Rejecting the null hypothesis implies that the larger model fits better.
}
}
\examples{
data("nhanes3lead")

fit <- lm(Math ~ poly(logBLL, 5) *
            (Male + Age + Race + PIR +
               Enough_Food),
          data = nhanes3lead)

# ADRF of logBLL on Math, unconditional
# inference
adrf1 <- adrf(fit, treat = "logBLL")

# Linear projection is sufficient for
# characterizing the ADRF
summary(adrf1, hypothesis = "linear")

# Compute the linear projection
proj1 <- curve_projection(adrf1, "linear")
# proj1 <- curve_projection(adrf1, ~logBLL) #same model

proj1

# Coefficients of the projection model
coef(proj1)
summary(proj1)

# Plot the projection
plot(proj1)

# Plot the projection over the ADRF
plot(adrf1, proj = proj1)

# Compute a cubic projection
proj2 <- curve_projection(adrf1, "cubic")
# proj2 <- curve_projection(adrf1, ~poly(logBLL, 3)) #same model

# Compare cubic to linear projection
anova(proj2, proj1)
}
\references{
Neugebauer, R., & van der Laan, M. (2007). Nonparametric causal effects based on marginal structural models. \emph{Journal of Statistical Planning and Inference}, 137(2), 419–434. \doi{10.1016/j.jspi.2005.12.008}
}
\seealso{
\itemize{
\item \code{\link[=plot.effect_curve]{plot.effect_curve()}} for plotting the effect curve and its projection
\item \code{\link{effect_curve}} for computing point estimates along the effect curve projection
\item \code{\link[=summary.effect_curve]{summary.effect_curve()}} for testing hypotheses about the effect curve, such as whether a given projection is sufficient
\item \code{\link[=anova]{anova()}} for comparing linear models
}
}
