#' Advanced Frequentist Methods for Changepoint Detection
#'
#' State-of-the-art algorithms including FPOP, Kernel CPD, and
#' Sparse Projection methods for high-dimensional data.
#'
#' @name advanced-methods
#' @noRd
#' @references
#' Maidstone et al. (2017) On Optimal Multiple Changepoint Algorithms
#'
#' Arlot et al. (2019) A Kernel Multiple Change-Point Algorithm
#'
#' Wang and Bhattacharjee (2021) High-Dimensional Changepoint Estimation
NULL

#' FPOP - Functional Pruning Optimal Partitioning
#'
#' More efficient than PELT for certain data types by maintaining
#' piecewise quadratic cost functions instead of just minimum values.
#' Achieves O(n) complexity in practice.
#'
#' @param data Numeric vector of observations
#' @param penalty Penalty for adding a changepoint. Can be:
#'   \itemize{
#'     \item \code{"bic"}: log(n)
#'     \item \code{"aic"}: 2
#'     \item \code{"mbic"}: 3*log(n)
#'     \item numeric: Custom penalty value
#'   }
#' @param min_segment Minimum segment length (default: 2)
#' @param cost_type Type of cost function:
#'   \itemize{
#'     \item \code{"mean"}: Gaussian mean change (default)
#'     \item \code{"meanvar"}: Gaussian mean and variance change
#'     \item \code{"poisson"}: Poisson rate change
#'   }
#'
#' @return List with:
#'   \item{changepoints}{Vector of changepoint locations}
#'   \item{cost}{Final optimal cost}
#'   \item{n_candidates}{Average candidates per time point (efficiency metric)}
#'
#' @export
#'
#' @references
#' Maidstone, R., Hocking, T., Rigaill, G., and Fearnhead, P. (2017).
#' On optimal multiple changepoint algorithms for large data.
#' Statistics and Computing, 27(2), 519-533.
#'
#' @examples
#' data <- c(rnorm(100, 0), rnorm(100, 3), rnorm(100, 1))
#' result <- fpop_detect(data, penalty = "bic")
#' print(result$changepoints)
fpop_detect <- function(data, penalty = "bic", min_segment = 2,
                        cost_type = "mean") {
  n <- length(data)
  
  if (n < 2 * min_segment) {
    return(list(changepoints = integer(0), cost = 0, n_candidates = 0))
  }
  
  pen <- switch(as.character(penalty),
                "bic" = log(n),
                "aic" = 2,
                "mbic" = 3 * log(n),
                "mdl" = 0.5 * log(n),
                as.numeric(penalty)
  )
  
  cumsum_data <- cumsum(data)
  cumsum_sq <- cumsum(data^2)
  
  compute_cost <- switch(cost_type,
                         "mean" = function(start, stop) {
                           if (stop < start) return(0)
                           len <- stop - start + 1
                           if (len < 1) return(0)
                           
                           if (start == 1) {
                             sum_x <- cumsum_data[stop]
                             sum_x2 <- cumsum_sq[stop]
                           } else {
                             sum_x <- cumsum_data[stop] - cumsum_data[start - 1]
                             sum_x2 <- cumsum_sq[stop] - cumsum_sq[start - 1]
                           }
                           
                           max(0, sum_x2 - sum_x^2 / len)
                         },
                         
                         "meanvar" = function(start, stop) {
                           len <- stop - start + 1
                           if (len < 2) return(0)
                           
                           if (start == 1) {
                             sum_x <- cumsum_data[stop]
                             sum_x2 <- cumsum_sq[stop]
                           } else {
                             sum_x <- cumsum_data[stop] - cumsum_data[start - 1]
                             sum_x2 <- cumsum_sq[stop] - cumsum_sq[start - 1]
                           }
                           
                           mu <- sum_x / len
                           sigma2 <- (sum_x2 - len * mu^2) / len
                           if (sigma2 <= 0) sigma2 <- 1e-10
                           
                           len * log(sigma2)
                         },
                         
                         "poisson" = function(start, stop) {
                           len <- stop - start + 1
                           if (len < 1) return(0)
                           
                           if (start == 1) {
                             sum_x <- cumsum_data[stop]
                           } else {
                             sum_x <- cumsum_data[stop] - cumsum_data[start - 1]
                           }
                           
                           if (sum_x <= 0) return(0)
                           lambda <- sum_x / len
                           
                           -sum_x * log(lambda) + len * lambda
                         }
  )
  
  get_cost_coeffs <- function(start, stop) {
    len <- stop - start + 1
    
    if (start == 1) {
      sum_x <- cumsum_data[stop]
      sum_x2 <- cumsum_sq[stop]
    } else {
      sum_x <- cumsum_data[stop] - cumsum_data[start - 1]
      sum_x2 <- cumsum_sq[stop] - cumsum_sq[start - 1]
    }
    
    list(a = len, b = -2 * sum_x, c = sum_x2)
  }
  
  F <- rep(Inf, n + 1)
  F[1] <- -pen
  
  cp_prev <- integer(n + 1)
  cp_prev[1] <- 0
  
  candidates <- list(list(tau = 0, left = -Inf, right = Inf))
  
  total_candidates <- 0
  
  for (t in min_segment:n) {
    best_cost <- Inf
    best_tau <- 0
    
    valid_candidates <- list()
    
    for (cand in candidates) {
      s <- cand$tau + 1
      
      if (t - s + 1 >= min_segment) {
        coeffs <- get_cost_coeffs(s, t)
        
        if (coeffs$a > 0) {
          mu_opt <- -coeffs$b / (2 * coeffs$a)
        } else {
          mu_opt <- 0
        }
        
        segment_cost <- coeffs$a * mu_opt^2 + coeffs$b * mu_opt + coeffs$c
        total_cost <- F[s] + segment_cost + pen
        
        if (total_cost < best_cost) {
          best_cost <- total_cost
          best_tau <- s - 1
        }
        
        if (total_cost <= best_cost + pen) {
          valid_candidates <- c(valid_candidates, list(cand))
        }
      } else {
        valid_candidates <- c(valid_candidates, list(cand))
      }
    }
    
    F[t + 1] <- best_cost
    cp_prev[t + 1] <- best_tau
    
    valid_candidates <- c(valid_candidates,
                          list(list(tau = t, left = -Inf, right = Inf)))
    
    candidates <- valid_candidates
    total_candidates <- total_candidates + length(candidates)
  }
  
  changepoints <- integer(0)
  t <- n + 1
  
  while (t > 1) {
    prev <- cp_prev[t]
    if (prev > 0) {
      changepoints <- c(prev, changepoints)
    }
    t <- prev + 1
  }
  
  list(
    changepoints = changepoints,
    cost = F[n + 1],
    n_candidates = total_candidates / (n - min_segment + 1)
  )
}

#' Kernel-based Changepoint Detection
#'
#' Detects changepoints using Maximum Mean Discrepancy (MMD) in a
#' Reproducing Kernel Hilbert Space. This nonparametric approach can
#' detect changes in any aspect of the distribution.
#'
#' @param data Numeric vector or matrix (rows = observations)
#' @param penalty Penalty for changepoints (default: "bic")
#' @param kernel Kernel function to use:
#'   \itemize{
#'     \item \code{"rbf"}: Radial Basis Function (Gaussian) kernel
#'     \item \code{"linear"}: Linear kernel
#'     \item \code{"poly"}: Polynomial kernel
#'     \item \code{"laplacian"}: Laplacian kernel
#'   }
#' @param bandwidth Kernel bandwidth (0 = automatic median heuristic)
#' @param min_segment Minimum segment length
#'
#' @return List with changepoints and kernel statistics
#'
#' @export
#'
#' @references
#' Arlot, S., Celisse, A., and Harchaoui, Z. (2019). A kernel multiple
#' change-point algorithm via model selection. Journal of Machine
#' Learning Research, 20(162), 1-56.
kernel_cpd_detect <- function(data, penalty = "bic", kernel = "rbf",
                              bandwidth = 0, min_segment = 5) {
  if (is.vector(data)) {
    data <- matrix(data, ncol = 1)
  }
  n <- nrow(data)
  d <- ncol(data)
  
  if (n < 2 * min_segment) {
    return(list(changepoints = integer(0), cost = 0))
  }
  
  pen <- switch(as.character(penalty),
                "bic" = log(n) * d,
                "aic" = 2 * d,
                "mbic" = 3 * log(n) * d,
                as.numeric(penalty)
  )
  
  if (bandwidth == 0) {
    sample_size <- min(200, n)
    idx <- sample(n, sample_size)
    distances <- as.vector(dist(data[idx, , drop = FALSE]))
    bandwidth <- median(distances)
    if (bandwidth == 0) bandwidth <- 1
  }
  
  kernel_func <- switch(kernel,
                        "rbf" = function(x, y) {
                          exp(-sum((x - y)^2) / (2 * bandwidth^2))
                        },
                        "linear" = function(x, y) {
                          sum(x * y)
                        },
                        "poly" = function(x, y) {
                          (1 + sum(x * y))^3
                        },
                        "laplacian" = function(x, y) {
                          exp(-sum(abs(x - y)) / bandwidth)
                        }
  )
  
  if (n <= 500) {
    K <- matrix(0, n, n)
    for (i in 1:n) {
      for (j in i:n) {
        K[i, j] <- kernel_func(data[i, ], data[j, ])
        K[j, i] <- K[i, j]
      }
    }
    
    cumsum_K <- matrix(0, n + 1, n + 1)
    for (i in 1:n) {
      for (j in 1:n) {
        cumsum_K[i + 1, j + 1] <- K[i, j] + cumsum_K[i, j + 1] +
          cumsum_K[i + 1, j] - cumsum_K[i, j]
      }
    }
    
    segment_cost <- function(start, stop) {
      len <- stop - start + 1
      if (len < min_segment) return(Inf)
      
      sum_K <- cumsum_K[stop + 1, stop + 1] -
        cumsum_K[start, stop + 1] -
        cumsum_K[stop + 1, start] +
        cumsum_K[start, start]
      
      -sum_K / len^2
    }
  } else {
    segment_cost <- function(start, stop) {
      len <- stop - start + 1
      if (len < min_segment) return(Inf)
      
      sum_K <- 0
      for (i in start:stop) {
        for (j in start:stop) {
          sum_K <- sum_K + kernel_func(data[i, ], data[j, ])
        }
      }
      
      -sum_K / len^2
    }
  }
  
  F <- rep(Inf, n + 1)
  F[1] <- -pen
  
  cp_prev <- integer(n + 1)
  cp_prev[1] <- 0
  
  candidates <- 0
  
  for (t in min_segment:n) {
    new_candidates <- integer(0)
    
    for (s in candidates) {
      if (t - s >= min_segment) {
        cost <- segment_cost(s + 1, t)
        total_cost <- F[s + 1] + cost + pen
        
        if (total_cost < F[t + 1]) {
          F[t + 1] <- total_cost
          cp_prev[t + 1] <- s
        }
        
        if (F[s + 1] + cost <= F[t + 1]) {
          new_candidates <- c(new_candidates, s)
        }
      } else {
        new_candidates <- c(new_candidates, s)
      }
    }
    
    candidates <- c(new_candidates, t)
  }
  
  changepoints <- integer(0)
  t <- n + 1
  
  while (t > 1) {
    prev <- cp_prev[t]
    if (prev > 0) {
      changepoints <- c(prev, changepoints)
    }
    t <- prev + 1
  }
  
  list(
    changepoints = changepoints,
    cost = F[n + 1],
    kernel = kernel,
    bandwidth = bandwidth
  )
}

#' Sparse Projection Changepoint Detection
#'
#' Detects changepoints in high-dimensional data using random sparse
#' projections to reduce dimensionality while preserving changepoint
#' structure.
#'
#' @param data Matrix with rows = observations, columns = dimensions
#' @param n_projections Number of random projections (default: 10)
#' @param penalty Penalty for changepoints
#' @param min_segment Minimum segment length
#' @param sparsity Sparsity level for projections (fraction of non-zeros)
#'
#' @return List with changepoints and projection information
#'
#' @export
#'
#' @references
#' Wang, D. and Bhattacharjee, M. (2021). High-dimensional changepoint
#' estimation via sparse projection. arXiv preprint.
sparse_projection_cpd <- function(data, n_projections = 10, penalty = "bic",
                                  min_segment = 5, sparsity = 0.3) {
  if (is.vector(data)) {
    data <- matrix(data, ncol = 1)
  }
  
  n <- nrow(data)
  d <- ncol(data)
  
  if (n < 2 * min_segment) {
    return(list(changepoints = integer(0)))
  }
  
  generate_sparse_projection <- function() {
    proj <- rep(0, d)
    n_nonzero <- max(1, ceiling(sparsity * d))
    idx <- sample(d, n_nonzero)
    proj[idx] <- sample(c(-1, 1), n_nonzero, replace = TRUE)
    proj / sqrt(n_nonzero)
  }
  
  projections <- replicate(n_projections, generate_sparse_projection())
  
  projected_data <- data %*% projections
  
  all_changepoints <- list()
  
  for (j in 1:n_projections) {
    result <- pelt(projected_data[, j], penalty = penalty,
                   min_segment = min_segment)
    all_changepoints[[j]] <- result$changepoints
  }
  
  all_cps <- unlist(all_changepoints)
  
  if (length(all_cps) == 0) {
    return(list(
      changepoints = integer(0),
      projections_used = n_projections,
      sparsity = sparsity
    ))
  }
  
  aggregated <- integer(0)
  threshold <- min_segment
  
  cp_counts <- table(all_cps)
  cp_locations <- as.integer(names(cp_counts))
  cp_freq <- as.integer(cp_counts)
  
  ord <- order(cp_freq, decreasing = TRUE)
  cp_locations <- cp_locations[ord]
  cp_freq <- cp_freq[ord]
  
  used <- rep(FALSE, length(cp_locations))
  
  for (i in seq_along(cp_locations)) {
    if (!used[i]) {
      loc <- cp_locations[i]
      freq <- cp_freq[i]
      
      if (freq >= max(2, n_projections / 3)) {
        aggregated <- c(aggregated, loc)
        
        nearby <- which(abs(cp_locations - loc) <= threshold)
        used[nearby] <- TRUE
      }
    }
  }
  
  list(
    changepoints = sort(aggregated),
    projections_used = n_projections,
    sparsity = sparsity,
    individual_results = all_changepoints
  )
}

#' E-Divisive Changepoint Detection
#'
#' Nonparametric changepoint detection using energy statistics.
#' Can detect changes in any aspect of the distribution.
#'
#' @param data Numeric vector or matrix
#' @param min_segment Minimum segment length
#' @param alpha Significance level for permutation tests
#' @param n_perm Number of permutations
#'
#' @return List with changepoints and test statistics
#'
#' @export
#'
#' @references
#' Matteson, D. S., and James, N. A. (2014). A nonparametric approach for
#' multiple change point analysis of multivariate data. Journal of the
#' American Statistical Association, 109(505), 334-345.
edivisive_detect <- function(data, min_segment = 5, alpha = 0.05,
                             n_perm = 199) {
  if (is.vector(data)) {
    data <- matrix(data, ncol = 1)
  }
  
  n <- nrow(data)
  d <- ncol(data)
  
  if (n < 2 * min_segment) {
    return(list(changepoints = integer(0)))
  }
  
  euclidean_dist <- function(x, y) {
    sqrt(sum((x - y)^2))
  }
  energy_statistic <- function(start, stop, split, data_matrix) {
    n1 <- split - start + 1
    n2 <- stop - split
    
    if (n1 < min_segment || n2 < min_segment) return(0)
    
    sum_xy <- 0
    sum_xx <- 0
    sum_yy <- 0
    
    idx1 <- start:split
    idx2 <- (split + 1):stop
    
    if (n1 > 50) idx1 <- sample(idx1, 50)
    if (n2 > 50) idx2 <- sample(idx2, 50)
    
    n1_eff <- length(idx1)
    n2_eff <- length(idx2)
    
    for (i in idx1) {
      for (j in idx2) {
        sum_xy <- sum_xy + euclidean_dist(data_matrix[i, ], data_matrix[j, ])
      }
    }
    
    for (i in seq_along(idx1)) {
      for (j in seq_along(idx1)) {
        if (i != j) {
          sum_xx <- sum_xx + euclidean_dist(data_matrix[idx1[i], ], data_matrix[idx1[j], ])
        }
      }
    }
    
    for (i in seq_along(idx2)) {
      for (j in seq_along(idx2)) {
        if (i != j) {
          sum_yy <- sum_yy + euclidean_dist(data_matrix[idx2[i], ], data_matrix[idx2[j], ])
        }
      }
    }
    
    mean_xy <- sum_xy / (n1_eff * n2_eff)
    mean_xx <- if (n1_eff > 1) sum_xx / (n1_eff * (n1_eff - 1)) else 0
    mean_yy <- if (n2_eff > 1) sum_yy / (n2_eff * (n2_eff - 1)) else 0
    
    n1_eff * n2_eff / (n1_eff + n2_eff) * (2 * mean_xy - mean_xx - mean_yy)
  }
  
  find_best_split <- function(start, stop, data_matrix) {
    best_stat <- 0
    best_loc <- -1
    
    for (t in (start + min_segment - 1):(stop - min_segment)) {
      stat <- energy_statistic(start, stop, t, data_matrix)
      if (stat > best_stat) {
        best_stat <- stat
        best_loc <- t
      }
    }
    
    list(location = best_loc, statistic = best_stat)
  }
  
  permutation_test <- function(start, stop, observed_stat) {
    count <- 0
    segment_data <- data[start:stop, , drop = FALSE]
    segment_n <- nrow(segment_data)
    
    for (p in 1:n_perm) {
      perm <- sample(segment_n)
      data_perm <- data
      data_perm[start:stop, ] <- segment_data[perm, , drop = FALSE]
      
      best <- find_best_split(start, stop, data_perm)
      if (best$statistic >= observed_stat) {
        count <- count + 1
      }
    }
    
    (count + 1) / (n_perm + 1)
  }
  
  changepoints <- integer(0)
  statistics <- numeric(0)
  
  queue <- list(c(1, n))
  
  while (length(queue) > 0) {
    segment <- queue[[1]]
    queue <- queue[-1]
    
    start <- segment[1]
    stop <- segment[2]
    
    if (stop - start + 1 < 2 * min_segment) next
    
    best <- find_best_split(start, stop, data)
    
    if (best$location > 0 && best$statistic > 0) {
      p_value <- permutation_test(start, stop, best$statistic)
      
      if (p_value < alpha) {
        changepoints <- c(changepoints, best$location)
        statistics <- c(statistics, best$statistic)
        
        queue <- c(queue, list(c(start, best$location)))
        queue <- c(queue, list(c(best$location + 1, stop)))
      }
    }
  }
  
  ord <- order(changepoints)
  
  list(
    changepoints = changepoints[ord],
    statistics = statistics[ord],
    alpha = alpha,
    n_perm = n_perm
  )
}

#' NOT - Narrowest-Over-Threshold Changepoint Detection
#'
#' Detects changepoints by finding the narrowest intervals that contain
#' significant changes, providing excellent localization.
#'
#' @param data Numeric vector
#' @param threshold Detection threshold (NULL for automatic)
#' @param min_segment Minimum segment length
#' @param contrast_type Type of contrast function
#'
#' @return List with changepoints and interval information
#'
#' @export
#'
#' @references
#' Baranowski, R., Chen, Y., and Fryzlewicz, P. (2019). Narrowest-over-
#' threshold detection of multiple change points and change-point-like
#' features. Journal of the Royal Statistical Society Series B, 81(3), 649-672.
not_detect <- function(data, threshold = NULL, min_segment = 5,
                       contrast_type = "pcwsConst") {
  n <- length(data)
  
  if (n < 2 * min_segment) {
    return(list(changepoints = integer(0)))
  }
  
  if (is.null(threshold)) {
    threshold <- sqrt(2 * log(n))
  }
  
  cumsum_data <- cumsum(data)
  cumsum_sq <- cumsum(data^2)
  
  contrast <- function(s, t, e) {
    n_left <- t - s + 1
    n_right <- e - t
    n_total <- e - s + 1
    
    if (n_left < 1 || n_right < 1) return(0)
    
    if (s == 1) {
      sum_left <- cumsum_data[t]
    } else {
      sum_left <- cumsum_data[t] - cumsum_data[s - 1]
    }
    sum_right <- cumsum_data[e] - cumsum_data[t]
    
    mean_left <- sum_left / n_left
    mean_right <- sum_right / n_right
    
    sqrt(n_left * n_right / n_total) * abs(mean_left - mean_right)
  }
  
  exceeding_intervals <- list()
  
  for (s in 1:(n - 2 * min_segment + 1)) {
    for (e in (s + 2 * min_segment - 1):n) {
      max_contrast <- 0
      best_t <- -1
      
      for (t in (s + min_segment - 1):(e - min_segment)) {
        c <- contrast(s, t, e)
        if (c > max_contrast) {
          max_contrast <- c
          best_t <- t
        }
      }
      
      if (max_contrast > threshold && best_t > 0) {
        exceeding_intervals <- c(exceeding_intervals,
                                 list(list(s = s, e = e, t = best_t,
                                           contrast = max_contrast)))
      }
    }
    
    if (length(exceeding_intervals) > 10000) break
  }
  
  if (length(exceeding_intervals) == 0) {
    return(list(changepoints = integer(0), intervals = list()))
  }
  
  widths <- sapply(exceeding_intervals, function(x) x$e - x$s)
  exceeding_intervals <- exceeding_intervals[order(widths)]
  
  changepoints <- integer(0)
  used <- rep(FALSE, n)
  
  for (interval in exceeding_intervals) {
    t <- interval$t
    
    if (!any(used[max(1, t - min_segment):min(n, t + min_segment)])) {
      changepoints <- c(changepoints, t)
      used[max(1, t - min_segment):min(n, t + min_segment)] <- TRUE
    }
  }
  
  list(
    changepoints = sort(changepoints),
    threshold = threshold,
    n_intervals = length(exceeding_intervals)
  )
}

#' CROPS - Changepoints for a Range of Penalties
#'
#' Efficiently computes optimal segmentations for a range of penalty
#' values, useful for model selection.
#'
#' @param data Numeric vector
#' @param penalty_range Vector of length 2: c(min_penalty, max_penalty)
#' @param method Segmentation method ("pelt" or "fpop")
#' @param min_segment Minimum segment length
#'
#' @return List with:
#'   \item{penalties}{Penalty values tested}
#'   \item{n_changepoints}{Number of changepoints for each penalty}
#'   \item{changepoints}{List of changepoint vectors}
#'   \item{costs}{Optimal costs}
#'
#' @export
#'
#' @references
#' Haynes, K., Eckley, I. A., and Fearnhead, P. (2017). Computationally
#' efficient changepoint detection for a range of penalties. Journal
#' of Computational and Graphical Statistics, 26(1), 134-143.
crops_detect <- function(data, penalty_range = c(0.1, 100),
                         method = "pelt", min_segment = 2) {
  n <- length(data)
  
  detect_func <- switch(method,
                        "pelt" = function(pen) pelt(data, penalty = pen, min_segment = min_segment),
                        "fpop" = function(pen) fpop_detect(data, penalty = pen, min_segment = min_segment),
                        function(pen) pelt(data, penalty = pen, min_segment = min_segment)
  )
  
  beta_min <- penalty_range[1]
  beta_max <- penalty_range[2]
  
  result_min <- detect_func(beta_min)
  result_max <- detect_func(beta_max)
  
  all_results <- list(
    list(penalty = beta_min, result = result_min),
    list(penalty = beta_max, result = result_max)
  )
  
  queue <- list(c(beta_min, beta_max))
  
  while (length(queue) > 0) {
    interval <- queue[[1]]
    queue <- queue[-1]
    
    beta_l <- interval[1]
    beta_r <- interval[2]
    
    res_l <- NULL
    res_r <- NULL
    
    for (r in all_results) {
      if (abs(r$penalty - beta_l) < 1e-10) res_l <- r$result
      if (abs(r$penalty - beta_r) < 1e-10) res_r <- r$result
    }
    
    n_l <- length(res_l$changepoints)
    n_r <- length(res_r$changepoints)
    
    if (n_l == n_r) {
      next
    }
    
    if (n_l == n_r + 1) {
      next
    }
    
    beta_mid <- (res_l$cost - res_r$cost + beta_r * n_r - beta_l * n_l) /
      (n_r - n_l)
    
    if (beta_mid <= beta_l || beta_mid >= beta_r) {
      next
    }
    
    result_mid <- detect_func(beta_mid)
    all_results <- c(all_results,
                     list(list(penalty = beta_mid, result = result_mid)))
    
    queue <- c(queue, list(c(beta_l, beta_mid)))
    queue <- c(queue, list(c(beta_mid, beta_r)))
  }
  
  penalties <- sapply(all_results, function(x) x$penalty)
  ord <- order(penalties)
  all_results <- all_results[ord]
  
  list(
    penalties = sapply(all_results, function(x) x$penalty),
    n_changepoints = sapply(all_results, function(x) length(x$result$changepoints)),
    changepoints = lapply(all_results, function(x) x$result$changepoints),
    costs = sapply(all_results, function(x) x$result$cost)
  )
}