% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/00roxygen.R, R/mvgauss.R
\name{MVgauss}
\alias{MVgauss}
\alias{dmvnorm}
\alias{dgmrf}
\alias{dautoreg}
\alias{dseparable}
\alias{unstructured}
\title{Multivariate Gaussian densities}
\usage{
dmvnorm(x, mu = 0, Sigma, log = FALSE, scale = 1)

dgmrf(x, mu = 0, Q, log = FALSE, scale = 1)

dautoreg(x, mu = 0, phi, log = FALSE, scale = 1)

dseparable(...)

unstructured(k)
}
\arguments{
\item{x}{Density evaluation point}

\item{mu}{Mean parameter vector}

\item{Sigma}{Covariance matrix}

\item{log}{Logical; Return log density?}

\item{scale}{Extra scale parameter - see section 'Scaling'.}

\item{Q}{Sparse precision matrix}

\item{phi}{Autoregressive parameters}

\item{...}{Log densities}

\item{k}{Dimension}
}
\value{
Vector of densities.
}
\description{
Multivariate Gaussian densities
}
\details{
Multivariate normal density evaluation is done using \code{dmvnorm()}. This is meant for dense covariance matrices. If \emph{many evaluations} are needed for the \emph{same covariance matrix} please note that you can pass matrix arguments: When \code{x} is a matrix the density is applied to each row of \code{x} and the return value will be a vector (length = \code{nrow(x)}) of densities.

The function \code{dgmrf()} is essentially identical to \code{dmvnorm()} with the only difference that \code{dgmrf()} is specified via the \emph{precision} matrix (inverse covariance) assuming that this matrix is \emph{sparse}.

Autoregressive density evaluation is implemented for all orders via \code{dautoreg()} (including the simplest AR1).
We note that this variant is for a \emph{stationary}, \emph{mean zero} and \emph{variance one} process.
FIXME: Provide parameterization via partial correlations.

Separable extension can be constructed for an unlimited number of inputs. Each input must be a function returning a \emph{gaussian} \emph{mean zero} \strong{log} density. The output of \code{dseparable} is another \strong{log} density which can be evaluated for array arguments. For example \code{dseparable(f1,f2,f3)} takes as input a 3D array \code{x}. \code{f1} acts in 1st array dimension of \code{x}, \code{f2} in 2nd dimension and so on. In addition to \code{x}, parameters \code{mu} and \code{scale} can be supplied - see below.
}
\section{Functions}{
\itemize{
\item \code{dmvnorm()}: Multivariate normal distribution. \link{OSA-residuals} can be used for argument \code{x}.

\item \code{dgmrf()}: Multivariate normal distribution. OSA is \emph{not} implemented.

\item \code{dautoreg()}: Gaussian stationary mean zero AR(k) density

\item \code{dseparable()}: Separable extension of Gaussian log-densities

\item \code{unstructured()}: Helper to generate an unstructured correlation matrix to use with \code{dmvnorm}

}}
\section{Scaling}{

All the densities accept a \code{scale} argument which replaces \code{SCALE} and \code{VECSCALE} functionality of TMB.
Scaling is applied elementwise on the residual \code{x-mu}. This works as expected when \code{scale} is a \emph{scalar} or a \emph{vector} object of the same length as \code{x}.
In addition, \code{dmvnorm} and \code{dgmrf} can be scaled by a vector of length equal to the covariance/precision dimension. In this case the \code{scale} parameter is recycled by row to meet the special row-wise vectorization of these densities.
}

\section{Unstructured correlation}{

Replacement of \code{UNSTRUCTURED_CORR} functionality of TMB. Constuct object using \code{us <- unstructured(k)}.
Now \code{us} has two methods: \code{x <- us$parms()} gives the parameter vector used as input to the objective function, and \code{us$corr(x)} turns the parameter vector into an unstructured correlation matrix.
}

\examples{
func <- function(x, sd, parm, phi) {
   ## IID N(0, sd^2)
   f1 <- function(x)sum(dnorm(x, sd=sd, log=TRUE))
   Sigma <- diag(2) + parm
   ## MVNORM(0, Sigma)
   f2 <- function(x)dmvnorm(x, Sigma=Sigma, log=TRUE)
   ## AR(2) process
   f3 <- function(x)dautoreg(x, phi=phi, log=TRUE)
   ## Separable extension (implicit log=TRUE)
   -dseparable(f1, f2, f3)(x)
}
parameters <- list(x = array(0, c(10, 2, 10)), sd=2, parm=1, phi=c(.9, -.2))
obj <- MakeADFun(function(p)do.call(func, p), parameters, random="x")
## Check that density integrates to 1
obj$fn()
## Check that integral is independent of the outer parameters
obj$gr()
## Check that we can simulate from this density
s <- obj$simulate()
}
